<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;

/**
 * Executes chunked deletion of node and paragraph revisions and
 * records purge totals and space reclaimed estimate.
 *
 * Strategy:
 *  - Drive deletions from the working tables:
 *      * fastrev_par_delete(rid)
 *      * fastrev_node_delete(vid)
 *  - Use small chunks to limit lock contention and optionally sleep
 *    between chunks to reduce pressure on busy sites.
 *  - Delete from field revision tables first, then from core revision tables.
 *  - Uses StatementInterface::rowCount() on executed queries for accurate counts.
 *  - Estimates bytes freed as SUM( rows_deleted(table) * avg_row_size(table) )
 *    using information_schema.TABLES for avg row size.
 */
final class Purger {

  /**
   * Cache for average row sizes (bytes) per table to avoid repeated lookups.
   *
   * @var array<string,float>
   */
  private array $avgRowCache = [];

  /**
   * @param \Drupal\Core\Database\Connection $db
   *   Database connection used for purge statements.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger for diagnostic messages.
   * @param \Drupal\fast_revision_purge\Service\RevisionTableMap $map
   *   Table discovery helper for revision field tables.
   * @param \Drupal\fast_revision_purge\Service\DbPlatform $platform
   *   DB platform wrapper (reserved for future branching).
   * @param \Drupal\fast_revision_purge\Service\StatsStorage $stats
   *   Stats storage used to persist purge totals and space reclaimed.
   */
  public function __construct(
    private Connection $db,
    private LoggerInterface $logger,
    private RevisionTableMap $map,
    private DbPlatform $platform,
    private StatsStorage $stats,
  ) {}

  /**
   * Purge paragraphs first, then nodes (safer for FK-like relationships).
   *
   * Also computes a best-effort estimate of freed bytes and persists:
   *  - total_node_revisions_deleted
   *  - total_para_revisions_deleted
   *  - total_lb_revisions_deleted (rows deleted from LB rev field table)
   *  - space_freed_last_run / space_freed_total
   *  - last_purge_timestamp
   *
   * @param int $chunk
   *   Number of ids to delete per iteration.
   * @param int $sleepMs
   *   Milliseconds to sleep between chunks (0 for no sleep).
   *
   * @return \Drupal\fast_revision_purge\Service\PurgeResult
   *   Totals deleted for nodes and paragraphs.
   */
  public function purge(int $chunk, int $sleepMs): PurgeResult {
    $result = new PurgeResult();

    // Aggregators for this run.
    $bytesFreed = 0.0;
    $lbDeletedTotal = 0;

    // Purge paragraphs first.
    $result->paragraphRevisionsDeleted = $this->purgeParagraphs($chunk, $sleepMs, $bytesFreed);

    // Then nodes (and track LB rows deleted from LB field table).
    $result->nodeRevisionsDeleted = $this->purgeNodes($chunk, $sleepMs, $bytesFreed, $lbDeletedTotal);
    $result->layoutBuilderRowsDeleted = $lbDeletedTotal;

    // Persist run totals and space reclaimed (rounded to int).
    try {
      $this->stats->updateAfterPurge(
        $result->nodeRevisionsDeleted,
        $result->paragraphRevisionsDeleted,
        $result->layoutBuilderRowsDeleted,
        (int) round($bytesFreed)
      );
      $this->logger->info(
        'Purge persisted: node={n}, para={p}, lb_rows={lb}, bytes={b}',
        ['n' => $result->nodeRevisionsDeleted, 'p' => $result->paragraphRevisionsDeleted, 'lb' => $lbDeletedTotal, 'b' => (int) round($bytesFreed)]
      );
    }
    catch (\Throwable $e) {
      $this->logger->warning('Failed to persist purge stats: @msg', ['@msg' => $e->getMessage()]);
    }

    return $result;
  }

  /**
   * Delete paragraph revisions in chunks.
   *
   * Order of operations per chunk:
   *  1) Load up to $chunk rids from fastrev_par_delete.
   *  2) Delete from all paragraph revision field tables referencing revision_id.
   *  3) Delete from paragraph revision data table (if present).
   *  4) Delete from paragraph revision core table (counting rows).
   *  5) Remove those rids from fastrev_par_delete.
   *  6) Sleep (optional).
   *
   * @param int $chunk
   *   Max rows per loop.
   * @param int $sleepMs
   *   Sleep in ms between loops.
   * @param float $bytesFreed
   *   (in/out) Accumulator for estimated bytes freed this run.
   *
   * @return int
   *   Total revision rows deleted from the paragraph core revision table.
   */
  private function purgeParagraphs(int $chunk, int $sleepMs, float &$bytesFreed): int {
    [$rev, $revData] = $this->detectParagraphRevisionTables();
    if (!$rev) {
      return 0;
    }

    $rids = $this->db->select('fastrev_par_delete', 'd')
      ->fields('d', ['rid'])
      ->orderBy('rid')
      ->range(0, $chunk)
      ->execute()
      ->fetchCol();

    if (empty($rids)) {
      return 0;
    }

    foreach ($this->map->getParagraphRevisionFieldTables() as $t) {
      $deleted = $this->deleteIn($t, 'revision_id', $rids, TRUE);
      if ($deleted > 0) {
        $bytesFreed += $deleted * $this->avgRowSize($t);
      }
    }

    if ($revData && $this->db->schema()->tableExists($revData)) {
      $deleted = $this->deleteIn($revData, 'revision_id', $rids, TRUE);
      if ($deleted > 0) {
        $bytesFreed += $deleted * $this->avgRowSize($revData);
      }
    }

    $deletedCore = $this->deleteIn($rev, 'revision_id', $rids, TRUE);
    if ($deletedCore > 0) {
      $bytesFreed += $deletedCore * $this->avgRowSize($rev);
    }

    $this->deleteIn('fastrev_par_delete', 'rid', $rids);

    if ($sleepMs > 0) usleep($sleepMs * 1000);

    return $deletedCore;
  }

  /**
   * Delete node revisions in chunks.
   *
   * Order of operations per chunk:
   *  1) Load up to $chunk vids from fastrev_node_delete.
   *  2) Delete from all node revision field tables referencing revision_id.
   *  3) Delete from node_field_revision (vid).
   *  4) Delete from node_revision (counting rows).
   *  5) Remove those vids from fastrev_node_delete.
   *  6) Sleep (optional).
   *
   * @param int $chunk
   *   Max rows per loop.
   * @param int $sleepMs
   *   Sleep in ms between loops.
   * @param float $bytesFreed
   *   (in/out) Accumulator for estimated bytes freed this run.
   * @param int $lbDeletedTotal
   *   (in/out) Number of rows deleted from the Layout Builder revision field table.
   *
   * @return int
   *   Total revision rows deleted from node_revision.
   */
  private function purgeNodes(int $chunk, int $sleepMs, float &$bytesFreed, int &$lbDeletedTotal): int {
    // Load up to $chunk vids from the working table.
    $vids = $this->db->select('fastrev_node_delete', 'd')
      ->fields('d', ['vid'])
      ->orderBy('vid')
      ->range(0, $chunk)
      ->execute()
      ->fetchCol();

    if (empty($vids)) {
      return 0;
    }

    foreach ($this->map->getNodeRevisionFieldTables() as $t) {
      $deleted = $this->deleteIn($t, 'revision_id', $vids, TRUE);
      if ($deleted > 0) {
        $bytesFreed += $deleted * $this->avgRowSize($t);
        if ($this->isLayoutBuilderRevFieldTable($t)) {
          $lbDeletedTotal += $deleted;
        }
      }
    }

    $deletedNfr = $this->deleteIn('node_field_revision', 'vid', $vids, TRUE);
    if ($deletedNfr > 0) {
      $bytesFreed += $deletedNfr * $this->avgRowSize('node_field_revision');
    }

    $deletedCore = $this->deleteIn('node_revision', 'vid', $vids, TRUE);
    if ($deletedCore > 0) {
      $bytesFreed += $deletedCore * $this->avgRowSize('node_revision');
    }
    $this->deleteIn('fastrev_node_delete', 'vid', $vids);

    if ($sleepMs > 0) usleep($sleepMs * 1000);

    return $deletedCore;
  }

  /**
   * DELETE helper with IN() list and optional row counting.
   */
  private function deleteIn(string $table, string $field, array $ids, bool $count = FALSE): int {
    if (empty($ids)) return 0;
    $schema = $this->db->schema();
    if (!$schema->tableExists($table)) return 0;
    $placeholders = [];
    $args = [];
    foreach ($ids as $i => $id) {
      $ph = ":p{$i}";
      $placeholders[] = $ph;
      $args[$ph] = (int) $id;
    }
    $in = implode(',', $placeholders);
    $sql = "DELETE FROM {$table} WHERE {$field} IN ({$in})";

    if ($count) {
      $stmt = $this->db->prepareStatement($sql, [], TRUE);
      $stmt->execute($args);
      return (int) $stmt->rowCount();
    }
    else {
      $this->db->query($sql, $args);
      return 0;
    }
  }

  /**
   * Determine whether a given table is the Layout Builder node revision field table.
   */
  private function isLayoutBuilderRevFieldTable(string $table): bool {
    if ($table === 'node_revision__layout_builder__layout') {
      return TRUE;
    }
    // Defensive: ends with "__layout_builder__layout".
    return str_ends_with($table, '__layout_builder__layout');
  }

  /**
   * Average row size (bytes) from information_schema.TABLES, cached per table.
   */
  private function avgRowSize(string $table): float {
    if (isset($this->avgRowCache[$table])) {
      return $this->avgRowCache[$table];
    }
    $schemaName = $this->db->getConnectionOptions()['database'] ?? NULL;
    if (!$schemaName) {
      return $this->avgRowCache[$table] = 0.0;
    }
    $avg = $this->db->query("
      SELECT CASE WHEN TABLE_ROWS > 0
                  THEN (DATA_LENGTH + INDEX_LENGTH) / TABLE_ROWS
                  ELSE 0 END AS avg_row
      FROM information_schema.TABLES
      WHERE TABLE_SCHEMA = :s AND TABLE_NAME = :t
    ", [':s' => $schemaName, ':t' => $table])->fetchField();

    return $this->avgRowCache[$table] = ($avg !== FALSE) ? (float) $avg : 0.0;
  }

  /**
   * Detect paragraph revision + revision data tables across schemas.
   *
   * @return array{0:string|null,1:string|null}
   *   [rev, revData]
   *   rev:    'paragraph_revision' or 'paragraphs_item_revision' if present.
   *   revData:'paragraph_field_revision' or 'paragraphs_item_revision_field_data' if present; null otherwise.
   */
  private function detectParagraphRevisionTables(): array {
    $schema = $this->db->schema();

    // Try entity-definition first.
    $rev = $revData = null;
    try {
      $def = \Drupal::entityTypeManager()->getDefinition('paragraph', FALSE);
      if ($def) {
        $candRev = (string) ($def->get('revision_table') ?? '');
        $candRevData = (string) ($def->get('revision_data_table') ?? '');
        if ($candRev && $schema->tableExists($candRev)) {
          $rev = $candRev;
        }
        if ($candRevData && $schema->tableExists($candRevData)) {
          $revData = $candRevData;
        }
      }
    } catch (\Throwable $e) {
      // fall through to presence checks
    }

    // Fallbacks for legacy/new combos.
    if (!$rev) {
      if ($schema->tableExists('paragraph_revision')) {
        $rev = 'paragraph_revision';
      } elseif ($schema->tableExists('paragraphs_item_revision')) {
        $rev = 'paragraphs_item_revision';
      }
    }

    if (!$revData) {
      if ($schema->tableExists('paragraph_field_revision')) {
        $revData = 'paragraph_field_revision';
      } elseif ($schema->tableExists('paragraphs_item_revision_field_data')) {
        $revData = 'paragraphs_item_revision_field_data';
      }
    }

    return [$rev, $revData];
  }

}

/**
 * Lightweight DTO to return purge totals to callers.
 */
final class PurgeResult {
  /** @var int Number of node revisions deleted from node_revision. */
  public int $nodeRevisionsDeleted = 0;

  /** @var int Number of paragraph revisions deleted from paragraph rev table. */
  public int $paragraphRevisionsDeleted = 0;

  /** @var int Number of rows deleted from the LB node revision field table. */
  public int $layoutBuilderRowsDeleted = 0;
}
