<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Thin wrapper to read/write the singleton stats row (id = 1).
 */
final class StatsStorage {

  /**
   * Construct a new StatsStorage service.
   *
   * @param \Drupal\Core\Database\Connection $db
   *   Database connection used to read/write the fastrev_stats table.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Time service used to obtain the current request time for timestamps.
   */
  public function __construct(
    private readonly Connection $db,
    private readonly TimeInterface $time,
  ) {}

  /**
   * Retrieve the singleton stats row.
   *
   * If the row with id=1 does not exist, a default "empty" row is returned
   * (but not persisted).
   *
   * @return array<string,int|null>
   *   An associative array of stats fields, including:
   *   - id
   *   - total_node_revisions_deleted
   *   - total_para_revisions_deleted
   *   - total_lb_revisions_deleted
   *   - space_freed_last_run
   *   - space_freed_total
   *   - potential_claimable_space (NULL when never planned)
   *   - last_dryrun_timestamp (NULL when never run)
   *   - last_purge_timestamp (NULL when never run)
   */
  public function get(): array {
    $row = $this->db->select('fastrev_stats', 's')
      ->fields('s')
      ->condition('id', 1)
      ->execute()
      ->fetchAssoc();
    return $row ?: $this->emptyRow();
  }

  /**
   * Update the dry-run estimate and timestamp.
   *
   * Writes the potential reclaimable bytes reported by the last planning
   * (dry run) pass and records when that dry run occurred.
   *
   * @param int $potentialBytes
   *   Estimated bytes that could be reclaimed (from the dry run).
   * @param int|null $timestamp
   *   Optional UNIX timestamp for when the dry run occurred; when NULL the
   *   current request time is used.
   */
  public function updateAfterDryRun(int $potentialBytes, ?int $timestamp = NULL): void {
    $ts = $timestamp ?? $this->time->getRequestTime();
    $this->db->merge('fastrev_stats')
      ->key(['id' => 1])
      ->fields([
        'potential_claimable_space' => $potentialBytes,
        'last_dryrun_timestamp' => $ts,
      ])
      ->execute();
  }

  /**
   * Update cumulative purge statistics and (optionally) clear dry-run estimate.
   *
   * Increments lifetime counters for nodes, paragraphs, and Layout Builder
   * revisions deleted, records the bytes freed in the last run, and updates
   * the overall space_freed_total. Since the prior dry-run estimate is stale
   * after a purge, you may clear `potential_claimable_space` to NULL.
   *
   * @param int $nodesDeleted
   *   Number of node revisions deleted in the purge run.
   * @param int $parasDeleted
   *   Number of paragraph revisions deleted in the purge run.
   * @param int $lbRowsDeleted
   *   Number of Layout Builder revision field rows deleted in the purge run.
   * @param int $bytesFreed
   *   Estimated bytes freed during this purge run.
   * @param int|null $timestamp
   *   Optional UNIX timestamp to record as the purge time. Defaults to the
   *   current request time when NULL.
   * @param bool $clearPotential
   *   When TRUE, sets potential_claimable_space to NULL after purge.
   */
  public function updateAfterPurge(
    int $nodesDeleted,
    int $parasDeleted,
    int $lbRowsDeleted,
    int $bytesFreed,
    ?int $timestamp = NULL,
    bool $clearPotential = TRUE
  ): void {
    $ts = $timestamp ?? $this->time->getRequestTime();

    // Read current counters in a safe defaulting way.
    $row = $this->get();
    $totalNode = (int) ($row['total_node_revisions_deleted'] ?? 0) + $nodesDeleted;
    $totalPara = (int) ($row['total_para_revisions_deleted'] ?? 0) + $parasDeleted;
    $totalLb   = (int) ($row['total_lb_revisions_deleted'] ?? 0) + $lbRowsDeleted;
    $spaceTot  = (int) ($row['space_freed_total'] ?? 0) + max(0, $bytesFreed);

    $fields = [
      'total_node_revisions_deleted' => $totalNode,
      'total_para_revisions_deleted' => $totalPara,
      'total_lb_revisions_deleted'   => $totalLb,
      'space_freed_last_run'         => max(0, $bytesFreed),
      'space_freed_total'            => $spaceTot,
      'last_purge_timestamp'         => $ts,
    ];

    if ($clearPotential) {
      $fields['potential_claimable_space'] = NULL;
    }

    $this->db->merge('fastrev_stats')
      ->key(['id' => 1])
      ->fields($fields)
      ->execute();
  }

  /**
   * Default values for the stats row (when no record exists yet).
   *
   * @return array<string,int|null>
   *   Zeroed cumulative counters, but NULL for "never run" fields.
   */
  private function emptyRow(): array {
    return [
      'id' => 1,
      'total_node_revisions_deleted' => 0,
      'total_para_revisions_deleted' => 0,
      'total_lb_revisions_deleted' => 0,
      'space_freed_last_run' => 0,
      'space_freed_total' => 0,
      'potential_claimable_space' => NULL,
      'last_dryrun_timestamp' => NULL,
      'last_purge_timestamp' => NULL,
    ];
  }

}
