<?php

namespace Drupal\fast_revision_purge\Service;

use Drupal\Core\Database\Connection;

/**
 * Lightweight DB size/usage helper (MySQL/MariaDB).
 *
 * Notes:
 * - Uses INFORMATION_SCHEMA.TABLES to compute sizes and list largest tables.
 * - Requires privileges to read INFORMATION_SCHEMA in the connected database.
 * - Intended for MySQL/MariaDB; other drivers may not expose the same columns.
 */
class TableStats {

  /**
   * @param \Drupal\Core\Database\Connection $db
   *   Drupal Database API connection.
   */
  public function __construct(private Connection $db) {
  }

  /**
   * Total database size (data + indexes) for the active schema.
   *
   * @return int|null
   *   Total bytes, or NULL if schema cannot be determined.
   */
  public function getDatabaseSizeBytes(): ?int {
    $schema = $this->getSchema();
    if (!$schema) {
      return NULL;
    }
    $sql = "SELECT SUM(DATA_LENGTH + INDEX_LENGTH) AS total_bytes
      FROM information_schema.TABLES
      WHERE TABLE_SCHEMA = :schema";
    $bytes = $this->db->query($sql, [':schema' => $schema])->fetchField();
    return $bytes !== FALSE ? (int) $bytes : NULL;
  }

  /**
   * Top-N largest tables in the schema (by DATA_LENGTH + INDEX_LENGTH).
   *
   * @param int $limit
   *   Number of rows to return (minimum 1).
   *
   * @return array<int, array{name:string, bytes:int}>
   *   A list of tables with total size in bytes, sorted descending.
   */
  public function getTopTables(int $limit = 3): array {
    $schema = $this->getSchema();
    if (!$schema) {
      return [];
    }

    $limit = max(1, (int) $limit);

    $sql =
      "SELECT TABLE_NAME AS name, (DATA_LENGTH + INDEX_LENGTH) AS bytes
       FROM information_schema.TABLES
       WHERE TABLE_SCHEMA = :schema
       ORDER BY bytes DESC
       LIMIT $limit";

    $rows = $this->db->query($sql, [':schema' => $schema])->fetchAllAssoc('name');

    $out = [];
    foreach ($rows as $row) {
      $out[] = [
        'name' => (string) $row->name,
        'bytes' => (int) $row->bytes,
      ];
    }
    return $out;
  }

  /**
   * Size of a specific table (data + indexes).
   *
   * @param string $table
   *   Table name (unprefixed or actual name in the schema).
   *
   * @return int|null
   *   Bytes for that table, or NULL if not found/schema unknown.
   */
  public function getTableSizeBytes(string $table): ?int {
    $schema = $this->getSchema();
    if (!$schema) {
      return NULL;
    }
    $bytes = $this->db->query(
      "SELECT (DATA_LENGTH + INDEX_LENGTH)
       FROM information_schema.TABLES
       WHERE TABLE_SCHEMA = :s AND TABLE_NAME = :t",
      [':s' => $schema, ':t' => $table]
    )->fetchField();

    return $bytes !== FALSE ? (int) $bytes : NULL;
  }

  /**
   * Average row size in bytes for a table, based on information_schema.
   *
   * Returns 0.0 when TABLE_ROWS=0 or the table is not present in
   * information_schema (or on drivers that don't expose these columns).
   *
   * @param string $table
   *   Table name.
   *
   * @return float
   *   Average bytes per row for the table (0.0 if unknown).
   */
  public function getAverageRowSize(string $table): float {
    $schema = $this->getSchema();
    if (!$schema) {
      return 0.0;
    }
    $avg = $this->db->query(
      "SELECT CASE WHEN TABLE_ROWS > 0
                   THEN (DATA_LENGTH + INDEX_LENGTH) / TABLE_ROWS
                   ELSE 0 END AS avg_row
       FROM information_schema.TABLES
       WHERE TABLE_SCHEMA = :s AND TABLE_NAME = :t",
      [':s' => $schema, ':t' => $table]
    )->fetchField();

    return $avg !== FALSE ? (float) $avg : 0.0;
  }

  /**
   * Row count for a table using COUNT(*). Avoid on huge tables in hot paths.
   *
   * @param string $table
   *   Table name.
   *
   * @return int|null
   *   Number of rows, or NULL if the table doesn't exist.
   */
  public function getRowCount(string $table): ?int {
    $schemaApi = $this->db->schema();
    if (!$schemaApi->tableExists($table)) {
      return NULL;
    }
    $count = $this->db->query("SELECT COUNT(*) FROM `{$table}`")->fetchField();
    return $count !== FALSE ? (int) $count : NULL;
  }

  /**
   * Human-friendly byte formatter (binary units).
   *
   * @param int|null $bytes
   *   Raw byte count or NULL.
   *
   * @return string
   *   Formatted string, e.g. "12.34 MB" or "N/A" for NULL.
   */
  public static function humanBytes(?int $bytes): string {
    if ($bytes === NULL) return 'N/A';
    $units = ['B','KB','MB','GB','TB','PB'];
    $i = 0;
    $val = (float) $bytes;
    while ($val >= 1024 && $i < count($units) - 1) {
      $val /= 1024;
      $i++;
    }
    return number_format($val, 2) . ' ' . $units[$i];
  }

  /**
   * Current schema/database name from the active connection.
   *
   * @return string|null
   *   Schema name or NULL when unavailable.
   */
  private function getSchema(): ?string {
    return $this->db->getConnectionOptions()['database'] ?? NULL;
  }

}
