<?php

namespace Drupal\favicons\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\favicons\FaviconsGenerator;
use Drupal\favicons\Form\FaviconConfigForm;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Favicon controller.
 */
class FaviconsController extends ControllerBase {

  /**
   * File URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * Favicons generator service.
   *
   * @var \Drupal\favicons\FaviconsGenerator
   */
  protected FaviconsGenerator $faviconsGenerator;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   File URL generator.
   */
  public function __construct(ConfigFactoryInterface $configFactory, FileUrlGeneratorInterface $fileUrlGenerator, FaviconsGenerator $faviconsGenerator) {
    $this->configFactory = $configFactory;
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->faviconsGenerator = $faviconsGenerator;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('file_url_generator'),
      $container->get('favicons.generator'),
    );
  }

  /**
   * Generate site.webmanifest
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JSON encoded string.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function siteWebmanifest(): Response {
    $config = $this->configFactory->get('favicons.settings');
    $fid = $config->get('favicon') ?? 0;
    $icons = [];

    if ($fid === 0) {
      return new Response('', 404);
    }

    $derivatives = $this->faviconsGenerator->getDerivatives($fid);

    foreach ($derivatives['derivatives'] as $style => $image_uri) {
      $icons[$style] = [
        'src' => $this->fileUrlGenerator->generateAbsoluteString($image_uri),
        'sizes' => $style,
        'type' => 'image/png',
        'purpose' => 'maskable',
      ];
    }

    $src = [
      'name' => $config->get('name') ?? '',
      'short_name' => $config->get('short_name') ?? '',
      'icons' => [ $icons ],
      'theme_color' => '#' . $config->get('theme_color') ?? 'ffffff',
      'background_color' => '#' . $config->get('background_color') ?? 'ffffff',
      'display' => 'standalone',
    ];

    return new Response(Json::encode($src));
  }

  /**
   * Create "SVG" from uploaded PNG.
   *
   * @return Response
   *   SVG output or 404.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function faviconSvg() {
    $config = $this->configFactory->get('favicons.settings');
    $fid = $config->get('favicon') ?? 0;
    $error404 = new Response('', 404);

    if ($fid == 0) {
      return $error404;
    }

    $file = $this->entityTypeManager()->getStorage('file')->load($fid);

    $sizes = getimagesize($file->getFileUri());

    if (empty($sizes)) {
      return $error404;
    }

    $width = $sizes[0];
    $height = $sizes[1];

    $svg = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" xmlns:xlink="http://www.w3.org/1999/xlink" width="' . $width . '" height="' . $height . '" viewBox="0 0 ' . $width . ' ' . $height . '">'
      . '<image width="' . $width . '" height="' . $height . '" xlink:href="data:image/png;base64,' . base64_encode(file_get_contents($file->getFileUri())) . '"></image></svg>';

    return new Response($svg);
  }
}
