<?php

namespace Drupal\favicons;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Favicon generator service.
 */
class FaviconsGenerator implements ContainerFactoryPluginInterface {
  /**
   * Image styles.
   */
  const array IMAGE_STYLES = [
    '96x96' => ['rel' => 'icon',],
    '180x180' => ['rel' => 'apple-touch-icon',],
  ];

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   EntityTypeManager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Configuration factory.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MessengerInterface $messenger,
    protected ConfigFactoryInterface $configFactory,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
  ) {}

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('config.factory'),
      $container->get('file_url_generator'),
    );
  }


  /**
   * Generate icons.
   *
   * @param int $fid
   *   File ID.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function generateIcons(int $fid): void {
    $derivatives = $this->getDerivatives($fid);

    foreach ($derivatives['derivatives'] as $imageStyleId => $derivative) {
      $style = $this->entityTypeManager->getStorage('image_style')->load($imageStyleId);
      $style->createDerivative($derivatives['image_uri'], $derivative['uri']);
    }
  }

  /**
   * Get list of derivatives based on styles.
   *
   * @param int $fid
   *   File ID.
   *
   * @return array
   *   List of derivatives.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getDerivatives(int $fid): array {
    $file = $this->entityTypeManager->getStorage('file')->load($fid);
    $derivatives = ['image_uri' => $file->getFileUri(), 'derivatives' => []];

    $styles = $this->entityTypeManager->getStorage('image_style')->loadMultiple(array_keys(self::IMAGE_STYLES));
    foreach ($styles as $style) {
      $derivative_uri = $style->buildUri($derivatives['image_uri']);
      $derivatives['derivatives'][$style->id()] = ['uri' => $derivative_uri, 'rel' => self::IMAGE_STYLES[$style->id()]['rel']];
    }

    return $derivatives;
  }

  /**
   * Get all icons.
   *
   * @return array
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getIcons(): array {
    $icons = [];

    $config = $this->configFactory->get('favicons.settings');

    $fid = $config->get('favicon') ?? 0;

    if ($fid === 0) {
      return $icons;
    }

    $favicon = $this->entityTypeManager->getStorage('file')->load($fid);

    // Regular favicon.
    $icons[] = [
      'uri' => $this->fileUrlGenerator->generateAbsoluteString($favicon->getFileUri()),
      'rel' => 'icon',
    ];

    // Different sizes.
    $derivatives = $this->getDerivatives($fid);
    foreach ($derivatives['derivatives'] as $imageStyleId => $derivative) {
      $icons[$imageStyleId] = [
        'uri' => $this->fileUrlGenerator->generateAbsoluteString($derivative['uri']),
        'rel' => $derivative['rel'],
      ];
    }

    return $icons;
  }

}
