<?php

namespace Drupal\favicons\Form;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\favicons\FaviconsGenerator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Favicon configuration form.
 */
class FaviconConfigForm extends ConfigFormBase {

  /**
   * EntityTypeManager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * FileSystem.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Favicon generator.
   *
   * @var \Drupal\favicons\FaviconsGenerator
   */
  protected FaviconsGenerator $faviconsGenerator;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Configuration factory.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   File system.
   */
  public function __construct(ConfigFactoryInterface $configFactory, EntityTypeManagerInterface $entityTypeManager, FileSystemInterface $fileSystem, FaviconsGenerator $faviconsGenerator) {
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
    $this->fileSystem = $fileSystem;
    $this->faviconsGenerator = $faviconsGenerator;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new FaviconConfigForm(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('file_system'),
      $container->get('favicons.generator')
    );
  }

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return ['favicons.settings'];
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'favicon_config_form';
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('favicons.settings');

    // Site name
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#required' => TRUE,
      '#default_value' => $config->get('name'),
    ];

    $form['short_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Short name'),
      '#required' => TRUE,
      '#default_value' => $config->get('short_name'),
    ];

    $form['theme_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Theme color'),
      '#required' => TRUE,
      '#size' => 6,
      '#default_value' => $config->get('theme_color'),
      '#description' => $this->t('Fill in the hex value of the color, like FFFFFF for white.'),
      '#maxlength' => 6,
    ];

    $form['background_color'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Background color'),
      '#size' => 6,
      '#required' => TRUE,
      '#default_value' => $config->get('background_color'),
      '#description' => $this->t('Fill in the hex value of the color, like FFFFFF for white.'),
      '#maxlength' => 6,
    ];

    $form['favicon'] = [
      '#title' => $this->t('Favicon'),
      '#type' => 'managed_file',
      '#upload_location' => 'public://',
      '#upload_validators' => [
        'file_validate_extensions' => ['png'],
      ],
      '#default_value' => ['fids' => $config->get('favicon') ?? ''],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('favicons.settings');
    $values = $form_state->getValues();

    // Handle file upload
    $favicon = $form_state->getValue('favicon', 0);
    $fid = 0;
    if (isset($favicon[0]) && !empty($favicon[0])) {
      $file = $this->entityTypeManager->getStorage('file')->load($favicon[0]);

      if ($file->id() !== $config->get('favicon')) {
        $sourceUri = $this->fileSystem->realpath($file->getFileUri());
        $defaultSchema = $this->configFactory->get('system.file')
            ->get('default_scheme') . '://';
        $destinationUri = $defaultSchema . 'favicon.png';
        $destination = $this->fileSystem->realpath($destinationUri);
        $this->fileSystem->move($sourceUri, $destination, FileExists::Replace);
        $file->setFileUri($destinationUri);
        $file->setPermanent();
        $file->save();
      }
      $fid = $file->id();
    }

    // Save configuration.
    $this->configFactory->getEditable('favicons.settings')
      ->set('name', $values['name'])
      ->set('short_name', $values['short_name'])
      ->set('theme_color', $values['theme_color'])
      ->set('background_color', $values['background_color'])
      ->set('favicon', $fid)
      ->save();


    // Generate ico, png, svg
    try {
      $this->faviconsGenerator->generateIcons($fid);
    } catch (InvalidPluginDefinitionException $e) {
      $this->messenger()->addError($e->getMessage());
    }

    parent::submitForm($form, $form_state);
  }


}
