<?php

namespace Drupal\featureflags_extensions\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\featureflags_extensions\FeatureFlagsExtensionsServiceInterface;
use Drupal\user\PermissionHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for the featureflags permissions form.
 */
class PermissionsForm extends FeatureFlagsExtensionsForm {

  /**
   * The permission handler.
   *
   * @var \Drupal\user\PermissionHandlerInterface
   */
  private PermissionHandlerInterface $permissionHandler;

  public function __construct(EntityTypeManagerInterface $entity_type_manager, PermissionHandlerInterface $permission_handler, FeatureFlagsExtensionsServiceInterface $featureflags_extensions) {
    parent::__construct($entity_type_manager, $featureflags_extensions);
    $this->permissionHandler = $permission_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('user.permissions'),
          $container->get('featureflags_extensions.service')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'featureflags_extensions_permissions_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $permissions = $this->permissionHandler->getPermissions();
    $config = $this->getConfigEntity();
    $options = [];
    foreach ($permissions as $id => $permission) {
      $options[str_replace('.', '__', $id)] = $permission['title'];
    }

    $form['permissions'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Permissions to deactivate'),
      '#options' => $options,
      '#default_value' => $config->getPermissions(),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $permissions = $form_state->getValue('permissions');
    $config = $this->getConfigEntity();
    $config->setPermissions($permissions);
    $config->save();

    $flag = $this->getRouteMatch()->getParameter('featureflag');

    $this->messenger()->addStatus(
          $this->t(
              'Saved the %label feature flag.', [
                '%label' => $flag->label(),
              ]
          )
      );
    $form_state->setRedirectUrl($flag->toUrl('collection'));
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigEntityType() {
    return 'featureflags_permissions';
  }

}
