<?php

namespace Drupal\featureflags_extensions;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\PermissionChecker as PermissionCheckerBase;
use Drupal\Core\Session\PermissionCheckerInterface;
use Drupal\featureflags\Entity\FeatureFlag;

/**
 * Custom permission checker.
 */
class PermissionChecker extends PermissionCheckerBase {

  /**
   * List of permissions to deactivate.
   *
   * @var array
   */
  private array $permissionsToDeactivate;

  /**
   * {@inheritdoc}
   */
  public function hasPermission(string $permission, AccountInterface $account): bool {

    if (in_array($permission, $this->getPermissionsToDeactivate())) {
      return FALSE;
    }

    return $this->permission_checker->hasPermission($permission, $account);
  }

  public function __construct(private PermissionCheckerInterface $permission_checker, protected EntityTypeManagerInterface $entityTypeManager, protected FeatureFlagsExtensionsServiceInterface $featureFlagsExtensionsService) {
    parent::__construct($this->entityTypeManager);


  /**
   * Helper function for loading all permissions to override.
   */
  private function getPermissionsToDeactivate() {
    if (empty($this->permissionsToDeactivate)) {
      $permissions_to_deactivate = [];
      /**
* @var \Drupal\featureflags\Entity\FeatureFlagInterface[] $flags
*/
      $flags = FeatureFlag::loadMultiple();

      foreach ($flags as $flag) {
        if (!$flag->getState()) {
          $extension = $this->featureFlagsExtensionsService->getExtension($flag, 'featureflags_permissions');
          if (empty($extension)) {
            continue;
          }
          $permissions = $extension->getPermissions();
          if (!empty($permissions)) {
            $permissions_to_deactivate = array_merge($permissions_to_deactivate, array_filter($permissions));
          }
        }
      }

      foreach ($permissions_to_deactivate as $key => $permission) {
        $permissions_to_deactivate[$key] = str_replace('__', '.', $permission);
      }

      $this->permissionsToDeactivate = $permissions_to_deactivate;
    }
    return $this->permissionsToDeactivate;
  }

}
