# Feeds Enhanced - Token Support

Provides universal token expansion for all Feed and Feed type text fields and
configurations.

## Overview

This submodule provides automatic token expansion for **any Drupal token** in all
Feeds and Feed type text fields and configurations. Any string containing `[token]` syntax will
be expanded during import, with no configuration required.

The expansion works universally across:
- **Feed entities**: source field, label, custom fields, and all text values
- **Feed Type configurations**: fetcher, parser, and processor plugin settings

Once enabled, all tokens are automatically expanded during every feed import.

## Features

* **Universal token expansion** - Any token in any Feed or Feed Type text field
* **Always active** - No configuration needed, just enable the module
* **Automatic integration** - Hooks into Feeds import pipeline via events
* **Performance optimized** - Session-based caching prevents redundant expansion
* **Community need** - Addresses Drupal.org issues [#3131079](https://www.drupal.org/project/feeds/issues/3131079) and [#3282260](https://www.drupal.org/project/feeds/issues/3282260)

## Requirements

* Drupal 10.1+ or Drupal 11
* Feeds module
* Token module

## Installation

Enable the module:
```bash
lando drush en feeds_enhanced_tokens
lando drush cr
```

## Usage

No configuration is required. Once enabled, any token in any Feed or Feed type text field will be
automatically expanded during feed imports.

### Common Token Types

The Token module provides a wide variety of token types. While any token will work,
here are some commonly used examples in Feeds configurations:

- **[feed:*]** - Feed entity data (id, label, source, etc.)
- **[feed-type:*]** - Feed type configuration
- **[current-user:*]** - User running the import (uid, name, mail, etc.)
- **[date:*]** - Current date/time (custom, short, medium, etc.)
- **[site:*]** - Site information (name, slogan, url, etc.)
- **[pantheon_si_tunnel:*]** - Pantheon SI tunnel ports (requires pantheon_si_tokens module)
- **Any custom tokens** - Tokens from other modules work automatically

### Example Use Cases

#### Dynamic SFTP Paths by Date

**Configuration**:
```
Fetcher directory: /exports/[date:custom:Y/m/d]
Filename pattern: import-[date:custom:His].csv
```

**Result**:
```
Directory: /exports/2025/12/21
Filename: import-153042.csv
```

#### Pantheon Secure Integration Tunnels

**Configuration**:
```
Feed source: 127.0.0.1:[pantheon_si_tunnel:pantheon_soip_ldap]/data.ldif
```

**Result**:
```
Feed source: 127.0.0.1:33061/data.ldif
```

The actual port number from the Pantheon SI tunnel is automatically injected.

#### User-Specific Feed Sources

**Configuration**:
```
Feed source: https://api.example.com/user-[current-user:uid]/feed.json
```

**Result**:
```
Feed source: https://api.example.com/user-42/feed.json
```

#### Dynamic Feed Labels

**Configuration**:
```
Feed label: Daily Import [date:custom:Y-m-d]
```

**Result**:
```
Feed label: Daily Import 2025-12-21
```

#### Processor Default Values with Tokens

**Configuration**:
```
Owner ID expression: [current-user:uid]
```

**Result**:
Imported nodes are owned by the user running the import.

### Programmatic Token Injection

This module enables programmatic token injection into any Feed or Feed Type field:

```php
// Set feed source with token programmatically
$feed = Feed::load(1);
$feed->setSource('127.0.0.1:[pantheon_si_tunnel:pantheon_soip_ldap]');
$feed->save();

// Tokens will be expanded automatically during import
$feed->import();
```

This approach makes feeds highly dynamic and configurable through code, the UI, or API.

## How It Works

1. **Event Subscriber**: Listens to `FeedsEvents::INIT_IMPORT` with high
   priority (1000)
2. **Field Traversal**: Iterates over all Feed entity fields and expands any
   string values
3. **Configuration Traversal**: Recursively traverses all plugin configurations
   (fetcher, parser, processor)
4. **Token Expansion**: Calls Token service to replace tokens with values
5. **Caching**: Session-based cache prevents redundant expansion of same
   token+context
6. **Cleanup**: Cache cleared after import completes
   (`FeedsEvents::IMPORT_FINISHED`)

## Performance

- **Early exit**: Quick bracket detection prevents unnecessary processing
- **Session caching**: Same token+context = cached result within single import
- **High priority**: Tokens expanded once before fetch, not repeatedly
- **Memory cleanup**: Cache cleared automatically after import
- **Minimal overhead**: Only processes strings that contain bracket syntax

## Token Context

During imports, the module provides the following context data for token replacement:

- **feed**: The current Feed entity
- **feed-type**: The Feed Type entity
- **current-user**: The user running the import (or cron user for scheduled imports)
- **date**: Current date/time
- **site**: Site configuration

Tokens from any module can access this context and will be expanded automatically.

## Troubleshooting

**Tokens not expanding**:
- Clear cache: `lando drush cr`
- Verify token syntax: Use `/admin/help/token` to browse available tokens
- Check Drupal logs: `lando drush wd-show --type=feeds_enhanced_tokens`

**Feed import fails after enabling**:
- Check for malformed tokens in Feed configurations
- Review Drupal logs: `lando drush wd-show --type=feeds_enhanced_tokens`
- Temporarily disable module to isolate issue

**Performance degradation**:
- Review token complexity (avoid nested or recursive tokens)
- Monitor cache clearing between imports
- Check for excessive token usage in large feed configurations

## Security Considerations

- Only expands tokens explicitly entered by users with Feed edit permissions
- Uses current user context for token evaluation (respects permissions)
- No automatic exposure of sensitive data
- Does not log actual token values for security

## Technical Details

### Event Priority

- **INIT_IMPORT @ 1000**: Very early execution ensures tokens expand before any
  plugin accesses configuration
- **IMPORT_FINISHED @ -1000**: Very late execution ensures cache cleanup after
  all processing

### Token Module Dependency

This module requires the Token module to function. Token expansion will not work
without it, as the module relies on Drupal's core token system for all
replacement operations.

### Field Iteration

The module uses `$feed->getFields()` to iterate over all entity fields,
ensuring universal coverage of:
- Base fields (source, label, etc.)
- Custom fields added to Feed entities
- Any contributed module fields

### Configuration Recursion

Plugin configurations are recursively traversed to find and expand tokens at
any nesting level:
- Fetcher configuration arrays
- Parser configuration arrays
- Processor configuration arrays
- Custom source definitions
- Mapping configurations

## Related Modules

- **Feeds** - Required base module
- **Token** - Required for token expansion
- **Pantheon SI Tokens** - Provides Pantheon Secure Integration tunnel tokens
- **Feeds Enhanced** - Parent module (convenient packaging location)

## Community Contribution

This module addresses active community feature requests:
- [Issue #3131079: Add token support for Http Fetcher source url](https://www.drupal.org/project/feeds/issues/3131079)
- [Issue #3282260: Make feed source URL a normal field, so that it supports
  tokens](https://www.drupal.org/project/feeds/issues/3282260)

The EventSubscriber approach was suggested by community members and refined for
universal application.

## License

This project is licensed under the GPL v2 or later.

## Maintainers

This module is maintained as part of the Feeds Enhanced project.
