<?php

declare(strict_types=1);

namespace Drupal\Tests\feeds_enhanced_tokens\Functional;

use Drupal\Tests\feeds\Functional\FeedsBrowserTestBase;

/**
 * Tests end-to-end token expansion in Feeds.
 *
 * @group feeds_enhanced_tokens
 */
class TokenExpansionFunctionalTest extends FeedsBrowserTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced_tokens',
    'token',
    'node',
    'user',
    'file',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests that tokens are expanded in feed source field.
   */
  public function testTokenExpansionInFeedSource(): void
  {
    $feed_type = $this->createFeedType([
      'fetcher' => 'http',
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);

    $feed = $this->createFeed($feed_type->id(), [
      'title' => 'Test Feed [current-user:uid]',
      'source' => 'http://example.com/feed-[current-user:uid].csv',
    ]);

    $feed_id = $feed->id();

    $this->drupalLogin($this->adminUser);

    $feed = $this->reloadEntity($feed);

    $this->assertEquals('Test Feed [current-user:uid]', $feed->label());

    $this->assertEquals(
      'http://example.com/feed-[current-user:uid].csv',
      $feed->getSource(),
    );
  }

  /**
   * Tests token service is available.
   */
  public function testTokenServiceAvailability(): void
  {
    $this->assertTrue(
      $this->container->has('feeds_enhanced_tokens.token_expander'),
    );

    $token_expander = $this->container->get(
      'feeds_enhanced_tokens.token_expander',
    );

    $this->assertInstanceOf(
      '\Drupal\feeds_enhanced_tokens\TokenExpander',
      $token_expander,
    );
  }

  /**
   * Tests event subscriber is registered.
   */
  public function testEventSubscriberRegistration(): void
  {
    $event_dispatcher = $this->container->get('event_dispatcher');
    $listeners = $event_dispatcher->getListeners(
      'feeds.init_import',
    );

    $found = FALSE;
    foreach ($listeners as $listener) {
      if (
        is_array($listener) &&
        $listener[0] instanceof
        \Drupal\feeds_enhanced_tokens\EventSubscriber\TokenExpansionSubscriber
      ) {
        $found = TRUE;
        break;
      }
    }

    $this->assertTrue(
      $found,
      'TokenExpansionSubscriber is registered for feeds.init_import',
    );
  }

}
