<?php

namespace Drupal\feeds_enhanced\Feeds\Fetcher\Form;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\feeds_enhanced\SftpClient;
use Drupal\key\KeyInterface;
use Drupal\key\KeyRepositoryInterface;

/**
 * Provides a trait for SFTP fetcher forms.
 *
 * This trait provides methods to create a base form for configuring SFTP
 * fetcher settings, including host, username, password, and timeout.
 */
trait SftpFetcherFormTrait {
  use StringTranslationTrait;

  /**
   * Returns a renderable array representing a form for configuring SFTP
   * fetcher settings.
   *
   * @param bool $as_required
   *   If TRUE, all elements will be defined as required.
   *
   * @return array[]
   *   A renderable array representing a form for configuring SFTP fetcher
   */
  protected function getBaseForm(bool $as_required = FALSE): array {
    return [
      'host' => [
        '#type' => 'textfield',
        '#title' => $this->t('Host'),
        '#description' => $this->t('The host to use for SFTP connections. Do not include any protocols(<em>https://, http://, sftp://, etc...</em>).'),
        '#default_value' => $this->getConfiguration('host') ?? '',
        '#placeholder' => $this->t('sftp.example.com'),
        '#required' => $as_required,
      ],
      'port' => [
        '#type' => 'number',
        '#title' => $this->t('Port'),
        '#description' => $this->t('The port to use for SFTP connections.'),
        '#default_value' => $this->getConfiguration('port')
          ?? SftpClient::defaultPort(),
        '#min' => 1,
        '#max' => 65535,
        '#required' => $as_required,
      ],
      'username' => [
        '#type' => 'textfield',
        '#title' => $this->t('User'),
        '#description' => $this->t('The username to use for SFTP connections.'),
        '#default_value' => $this->getConfiguration('username') ?? '',
        '#required' => $as_required,
      ],
      'password' => [
        '#type' => 'select',
        '#title' => $this->t('Password'),
        '#description' => $this->t('The password to use for SFTP connections. Configure passwords <a href="@url">here</a>.', [
          '@url' => $this->getKeysUrl(),
        ]),
        '#options' => $this->getKeyOptions(),
        '#default_value' => $this->getConfiguration('password') ?? '',
        '#required' => $as_required,
      ],
      'timeout' => [
        '#type' => 'number',
        '#title' => $this->t('Request timeout'),
        '#description' => $this->t('Timeout in seconds to wait for a SFTP request to finish.'),
        '#default_value' => $this->getConfiguration('timeout')
          ?? SftpClient::defaultTimeout(),
        '#min' => 0,
        '#required' => $as_required,
      ],
    ];
  }

  /**
   * Returns an array of Key names, keyed by their respective IDs.
   *
   * @return array
   *   An array of Key names, keyed by their respective IDs.
   */
  protected function getKeyOptions(): array {
    $options = array_map(
      fn (KeyInterface $key) => $key->label(),
      $this->keyRepository()->getKeys()
    );
    asort($options);
    return $options;
  }

  /**
   * Returns the URL to manage keys.
   *
   * @return string
   *   The URL to manage keys.
   */
  protected function getKeysUrl(): string {
    return Url::fromRoute('entity.key.collection')->toString();
  }

  /**
   * Returns the specified configuration value.
   *
   * @param string $config_name
   *   The name of the configuration to retrieve.
   *
   * @return array
   *   The current configuration.
   */
  abstract protected function getConfiguration(string $config_name): mixed;

  /**
   * Returns a reference to the key repository service.
   *
   * @return \Drupal\key\KeyRepositoryInterface
   *   A reference to the key repository service.
   */
  abstract protected function keyRepository(): KeyRepositoryInterface;

}
