<?php

namespace Drupal\Tests\feeds_enhanced\Functional;

use Drupal\feeds\Entity\Feed;
use Drupal\feeds\Entity\FeedType;
use Drupal\feeds_enhanced\SftpClient;
use Drupal\key\Entity\Key;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Test SFTP client that extends the real SftpClient for testing.
 */
class TestingSftpClient extends SftpClient {

  /**
   * The test file path to use instead of SFTP.
   *
   * @var string
   */
  protected string $testFilePath;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $data, string $testFilePath = '') {
    // Store test file path.
    $this->testFilePath = $testFilePath;

    // Store configuration without calling parent to avoid SFTP connection.
    $this->host = $data['host'] ?? '';
    $this->username = $data['username'] ?? '';
    $this->password = $data['password'] ?? '';

    // Don't create actual SFTP connection.
  }

  /**
   * {@inheritdoc}
   */
  public function test(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isLoggedIn(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function logIn(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($source, &$destination): bool {
    // Use test file instead of SFTP.
    if (empty($this->testFilePath) || !file_exists($this->testFilePath)) {
      return FALSE;
    }

    $content = file_get_contents($this->testFilePath);
    if ($content === FALSE) {
      return FALSE;
    }

    return file_put_contents($destination, $content) !== FALSE;
  }

  /**
   * Returns the decrypted password for testing verification.
   *
   * @return string
   */
  public function getDecryptedPassword(): string {
    return $this->password;
  }

}

/**
 * Tests SFTP fetcher functionality with Key integration.
 *
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
 * @group feeds_enhanced
 */
class SftpFetcherKeyIntegrationTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced',
    'key',
    'node',
    'user',
    'text',
    'filter',
  ];

  /**
   * The test CSV file path.
   *
   * @var string
   */
  protected $csvFile;

  /**
   * The test key entity.
   *
   * @var \Drupal\key\Entity\Key
   */
  protected $testKey;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create test CSV file with content.
    $this->csvFile = $this->createTestFile();
    $this->assertFileExists($this->csvFile);

    // Create a test key for SFTP password.
    $this->createTestKey();
  }

  /**
   * Creates a test CSV file in the public files directory.
   *
   * @return string
   *   The path to the created file.
   */
  protected function createTestFile() {
    $file_path = 'public://test_feed.csv';
    $csv_content = "title,body\nTest Article 1,This is the body of test article 1\nTest Article 2,This is the body of test article 2";
    file_put_contents($file_path, $csv_content);
    return $this->container->get('file_system')->realpath($file_path);
  }

  /**
   * Creates a test Key entity with a password.
   */
  protected function createTestKey() {
    $this->testKey = Key::create([
      'id' => 'test_sftp_password',
      'label' => 'Test SFTP Password',
      'key_type' => 'authentication',
      'key_provider' => 'config',
      'key_provider_settings' => [
        'key_value' => 'test_password_123',
      ],
    ]);
    $this->testKey->save();
  }

  /**
   * Tests the SFTP fetcher with Key integration.
   */
  public function testSftpFetcherWithKeyIntegration() {
    // Create a new feed type with Key configuration.
    $feed_type = FeedType::create([
      'id' => 'sftp_feed',
      'label' => 'SFTP Feed',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'example.com',
        'username' => 'testuser',
    // This references our Key ID.
        'password' => 'test_sftp_password',
        'timeout' => 90,
      ],
      'parser' => 'csv',
      'parser_configuration' => [
        'delimiter' => ',',
        'no_headers' => FALSE,
        'header_lines' => 1,
      ],
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => [
          'type' => 'article',
        ],
      ],
      'mappings' => [
        [
          'target' => 'title',
          'map' => ['value' => 'title'],
        ],
        [
          'target' => 'body',
          'map' => ['value' => 'body'],
          'settings' => ['format' => 'plain_text'],
        ],
      ],
    ]);
    $feed_type->save();

    // Create admin user.
    $adminUser = $this->drupalCreateUser([
      'access feed overview',
      'administer feeds',
      'create sftp_feed feeds',
      'import sftp_feed feeds',
      'create article content',
      'edit any article content',
      'delete any article content',
      'access administration pages',
      'bypass node access',
      'administer keys',
    ], NULL, TRUE);
    $adminUser->addRole('administrator');
    $adminUser->save();
    $this->drupalLogin($adminUser);

    // Create a feed.
    $feed = Feed::create([
      'title' => 'Test SFTP Feed',
      'type' => 'sftp_feed',
      'source' => 'sftp://example.com/test.csv',
    ]);
    $feed->save();

    // Override the SftpClient class temporarily for this test.
    $originalClass = SftpClient::class;
    $testFilePath = $this->csvFile;

    // Create a wrapper around the SftpFetcher to inject our test client.
    $fetcher = $feed_type->getFetcher();
    $reflection = new \ReflectionClass($fetcher);

    // Use reflection to test the getPw() method directly.
    $getPwMethod = $reflection->getMethod('getPw');
    $getPwMethod->setAccessible(TRUE);

    // Test that the Key integration works.
    $decryptedPassword = $getPwMethod->invoke($fetcher);
    $this->assertEquals('test_password_123', $decryptedPassword, 'Key integration failed - password not decrypted correctly');

    // Test the complete configuration flow.
    $getConfigMethod = $reflection->getMethod('getConfiguration');
    $getConfigMethod->setAccessible(TRUE);
    $config = $getConfigMethod->invoke($fetcher);

    // Verify the configuration contains our Key ID.
    $this->assertEquals('test_sftp_password', $config['password'], 'Configuration should contain Key ID');

    // Verify the Key entity exists and has correct value.
    $key = $this->testKey;
    $this->assertEquals('test_password_123', $key->getKeyValue(), 'Key entity should contain test password');

    // Test that a TestingSftpClient can be created with the decrypted password.
    $testConfig = [...$config, ...['password' => $decryptedPassword]];
    $testClient = new TestingSftpClient($testConfig, $this->csvFile);

    // Verify the client received the decrypted password.
    $this->assertEquals('test_password_123', $testClient->getDecryptedPassword(), 'SFTP client should receive decrypted password');

    // Test file retrieval.
    $tempFile = tempnam(sys_get_temp_dir(), 'sftp_test');
    $result = $testClient->getFile('test.csv', $tempFile);
    $this->assertTrue($result, 'File retrieval should succeed');

    // Verify file content.
    $content = file_get_contents($tempFile);
    $this->assertStringContainsString('Test Article 1', $content, 'Retrieved file should contain test data');

    // Clean up.
    unlink($tempFile);
  }

}
