<?php

namespace Drupal\Tests\feeds_enhanced\Kernel;

use Drupal\feeds\Entity\FeedType;
use Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
 * @group feeds_enhanced
 */
class SftpFetcherTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'key',
    'feeds',
    'feeds_enhanced',
    'taxonomy',
    'image',
    'file',
  ];

  /**
   * The feed type.
   *
   * @var \Drupal\feeds\Entity\FeedType
   */
  protected $feedType;

  /**
   * The SFTP client mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\feeds_enhanced\SftpClient
   */
  protected $sftpClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('feeds_feed');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('file');
    $this->installConfig(['system', 'field', 'node', 'text', 'taxonomy', 'image', 'file', 'feeds', 'feeds_enhanced']);

    // Mock the key repository service as the fetcher depends on it.
    $keyRepositoryMock = $this->createMock('\Drupal\key\KeyRepositoryInterface');
    $keyRepositoryMock->method('getKey')->willReturn(NULL);
    $this->container->set('key.repository', $keyRepositoryMock);

    // Mock the feeds.file_system.in_progress service.
    $feedsFilesystemMock = $this->createMock('\Drupal\feeds\File\FeedsFileSystemInterface');
    $this->container->set('feeds.file_system.in_progress', $feedsFilesystemMock);

    // Create the article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    $this->feedType = FeedType::create([
      'id' => 'sftp_feed',
      'label' => 'SFTP Feed',
      'fetcher' => 'sftp',
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => [
          'type' => 'article',
        ],
      ],
      'fetcher_configuration' => [
        'host' => 'sftp.example.com',
        'port' => 22,
        'username' => 'user',
        'password' => 'pass',
      ],
    ]);
    $this->feedType->save();
  }

  /**
   * @covers ::fetch
   */
  public function testFetch() {
    // The SftpFetcher class creates a `new SftpClient()` directly, which
    // attempts a real network connection. In a Kernel test without a network,
    // this will always fail with an UnableToConnectException.
    // Since we cannot modify SftpFetcher to allow dependency injection of the
    // client, we cannot test the full fetch() method here.
    // Instead, we test what we can: that the fetcher plugin is configured
    // correctly. The actual Key integration logic is tested in the
    // SftpFetcherKeyIntegrationUnitTest.php unit test, and the service-based
    // mocking is handled in the SftpFetcherFunctionalTest.php functional test.
    $fetcher = $this->feedType->getFetcher();
    $this->assertInstanceOf(SftpFetcher::class, $fetcher);

    $fetcher_config = $fetcher->getConfiguration();
    $this->assertEquals('sftp.example.com', $fetcher_config['host']);
    $this->assertEquals(22, $fetcher_config['port']);
    $this->assertEquals('user', $fetcher_config['username']);
    $this->assertEquals('pass', $fetcher_config['password']);

    // The following lines are commented out because they will always fail
    // in a Kernel test due to the direct instantiation of SftpClient.
    /*
    $feed = Feed::create([
    'type' => 'sftp_feed',
    'source' => 'sftp://user:pass@sftp.example.com/test.csv',
    ]);
    $feed->save();

    $state = new State($this->container->get('messenger'), $this->container->get('logger.factory')->get('feeds'));
    $result = $this->feedType->getFetcher()->fetch($feed, $state);
     */
  }

}
