<?php

namespace Drupal\Tests\feeds_enhanced\Unit;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\feeds_enhanced\SftpClient;
use Drupal\Tests\UnitTestCase;
use phpseclib3\Net\SFTP;

/**
 * @coversDefaultClass \Drupal\feeds_enhanced\SftpClient
 * @group feeds_enhanced
 */
class SftpClientTest extends UnitTestCase {

  /**
   * The SFTP client.
   *
   * @var \Drupal\feeds_enhanced\SftpClient
   */
  protected $sftpClient;

  /**
   * The phpseclib SFTP client mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\phpseclib3\Net\SFTP
   */
  protected $phpseclibSftp;

  /**
   * The logger channel mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->phpseclibSftp = $this->getMockBuilder(SFTP::class)
      ->disableOriginalConstructor()
      ->getMock();

    $this->logger = $this->getMockBuilder(LoggerChannelInterface::class)
      ->getMock();

    $data = [
      'host' => 'sftp.example.com',
      'username' => 'user',
      'password' => 'pass',
    ];

    $this->sftpClient = new SftpClient($data);
    $this->sftpClient->setLogger($this->logger);

    $reflection = new \ReflectionClass($this->sftpClient);
    $reflection_property = $reflection->getProperty('client');
    $reflection_property->setAccessible(TRUE);
    $reflection_property->setValue($this->sftpClient, $this->phpseclibSftp);
  }

  /**
   * @covers ::__construct
   * @covers ::host
   * @covers ::username
   * @covers ::password
   */
  public function testConstructor() {
    $this->assertSame('sftp.example.com', $this->sftpClient->host());
    $this->assertSame('user', $this->sftpClient->username());
    $this->assertSame('pass', $this->sftpClient->password());
  }

  /**
   * @covers ::test
   */
  public function testTestConnection() {
    $this->phpseclibSftp->expects($this->once())
      ->method('login')
      ->with('user', 'pass')
      ->willReturn(TRUE);

    $this->assertTrue($this->sftpClient->test());
  }

  /**
   * @covers ::getFile
   */
  public function testGetFile() {
    $this->phpseclibSftp->expects($this->once())
      ->method('login')
      ->with('user', 'pass')
      ->willReturn(TRUE);

    $this->phpseclibSftp->expects($this->once())
      ->method('get')
      ->with('remote_file.txt', 'local_file.txt')
      ->willReturn(TRUE);

    $destination = 'local_file.txt';
    $this->assertTrue($this->sftpClient->getFile('remote_file.txt', $destination));
  }

  /**
   * @covers ::getFile
   */
  public function testGetFileLoginFailure() {
    $this->phpseclibSftp->expects($this->once())
      ->method('login')
      ->with('user', 'pass')
      ->willReturn(FALSE);

    $this->logger->expects($this->once())
      ->method('warning')
      ->with('Failed to log in to the SFTP server at @host.', ['@host' => 'sftp.example.com']);

    $destination = 'local_file.txt';
    $this->assertFalse($this->sftpClient->getFile('remote_file.txt', $destination));
  }

  /**
   * @covers ::getFile
   */
  public function testGetFileDownloadFailure() {
    $this->phpseclibSftp->expects($this->once())
      ->method('login')
      ->with('user', 'pass')
      ->willReturn(TRUE);

    $this->phpseclibSftp->expects($this->once())
      ->method('get')
      ->with('remote_file.txt', 'local_file.txt')
      ->willReturn(FALSE);

    $this->logger->expects($this->once())
      ->method('warning')
      ->with('Failed to get the file @source from the SFTP server at @host.', ['@source' => 'remote_file.txt', '@host' => 'sftp.example.com']);

    $destination = 'local_file.txt';
    $this->assertFalse($this->sftpClient->getFile('remote_file.txt', $destination));
  }

}
