<?php

declare(strict_types=1);

namespace Drupal\feeds_enhanced_tokens;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Utility\Token;
use Drupal\feeds\FeedInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for expanding tokens in Feeds text fields and configuration.
 */
class TokenExpander
{

  /**
   * Token cache for current import session.
   *
   * @var array
   */
  protected array $expandedTokenCache = [];

  /**
   * Constructs a TokenExpander object.
   *
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected Token $token,
    protected AccountProxyInterface $currentUser,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerInterface $logger,
  ) {}

  /**
   * Expands tokens in a string value.
   *
   * @param string $text
   *   The text containing tokens to expand.
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity for token context.
   * @param array $additionalData
   *   Additional token data (optional).
   *
   * @return string
   *   The text with tokens expanded.
   */
  public function expandString(
    string $text,
    FeedInterface $feed,
    array $additionalData = [],
  ): string {
    if (!$this->containsTokens($text)) {
      return $text;
    }

    $cacheKey = md5($text . serialize($additionalData));
    if (array_key_exists($cacheKey, $this->expandedTokenCache)) {
      return $this->expandedTokenCache[$cacheKey];
    }

    $data = [
      ...$this->buildTokenContext($feed),
      ...$additionalData,
    ];

    $expanded = $this->token->replace(
      $text,
      $data,
      ['clear' => FALSE],
    );

    $this->expandedTokenCache[$cacheKey] = $expanded;
    return $expanded;
  }

  /**
   * Expands tokens in plugin configuration array.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity for token context.
   *
   * @return array
   *   Configuration with tokens expanded.
   */
  public function expandConfiguration(
    array $configuration,
    FeedInterface $feed,
  ): array {
    return $this->traverseAndExpand($configuration, $feed);
  }

  /**
   * Recursively traverses array and expands string values.
   *
   * @param mixed $value
   *   The value to process (array, string, or other).
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity for token context.
   *
   * @return mixed
   *   The processed value with tokens expanded.
   */
  protected function traverseAndExpand(
    mixed $value,
    FeedInterface $feed,
  ): mixed {
    if (is_array($value)) {
      return array_map(
        fn($item) => $this->traverseAndExpand($item, $feed),
        $value,
      );
    }

    if (is_string($value)) {
      return $this->expandString($value, $feed);
    }

    return $value;
  }

  /**
   * Checks if text contains token syntax.
   *
   * @param string $text
   *   The text to check.
   *
   * @return bool
   *   TRUE if text contains token syntax.
   */
  protected function containsTokens(string $text): bool
  {
    return str_contains($text, '[') && str_contains($text, ']');
  }

  /**
   * Builds token context data array.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity.
   *
   * @return array
   *   Token context data.
   */
  protected function buildTokenContext(FeedInterface $feed): array
  {
    return [
      'feed' => $feed,
      'feed-type' => $feed->getType(),
      'current-user' => $this->currentUser->getAccount(),
      'site' => NULL,
      'date' => NULL,
    ];
  }

  /**
   * Clears the token cache.
   *
   * Called at end of import to free memory.
   */
  public function clearCache(): void
  {
    $this->expandedTokenCache = [];
  }

}
