<?php

namespace Drupal\feeds_enhanced\Feeds\Fetcher\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Plugin\Type\ExternalPluginFormBase;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SftpFetcherForm
 *  The configuration form for SFTP fetchers to be displayed on the Feed
 *  edit page.
 */
class SftpFetcherFeedForm extends ExternalPluginFormBase implements ContainerInjectionInterface {
  use SftpFetcherFormTrait;

  /**
   * The current feed being edited.
   *
   * @var \Drupal\feeds\FeedInterface|null
   */
  protected ?FeedInterface $feed = NULL;

  /**
   * Constructs an SFTP Fetcher form object.
   *
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository.
   */
  public function __construct(
    protected KeyRepositoryInterface $keyRepository,
  ) {}

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container): static {
    return new self(
      $container->get('key.repository'),
    );
  }

  /**
   * {@inheritDoc}
   */
  protected function keyRepository(): KeyRepositoryInterface {
    return $this->keyRepository;
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(
    array $form,
    FormStateInterface $form_state,
    ?FeedInterface $feed = NULL,
  ): array {
    // Store internal reference for use in other methods.
    $this->feed = $feed;
    if ($feed && $feed->isNew()) {
      $feed->setConfigurationFor(
        $this->plugin,
        $this->plugin->getConfiguration(),
      );
    }
    return [
      'source' => [
        '#title' => $this->t('Remote file path'),
        '#type' => 'textfield',
        '#default_value' => $feed->getSource(),
        '#maxlength' => 2048,
        '#required' => TRUE,
      ],
      // Remainder of base form should follow after source.
      ...$this->getBaseForm(TRUE),
    ];
  }

  /**
   * {@inheritDoc}
   */
  protected function getConfiguration(string $config_name): mixed {
    $config = $this?->feed?->getConfigurationFor($this->plugin) ?? [];
    return $config[$config_name] ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(
    array &$form,
    FormStateInterface $form_state,
    ?FeedInterface $feed = NULL,
  ): void {
    $feed->setSource($form_state->getValue('source'));
    $host_value = $form_state->getValue('host');
    $parsed = $this->parseHostAndPort($host_value);
    $config = [
      ...$feed->getConfigurationFor($this->plugin),
      ...array_intersect_key(
        $form_state->getValues(),
        $this->getBaseForm(),
      ),
      'host' => $parsed['host'],
      'port' => $parsed['port'],
    ];
    $feed->setConfigurationFor($this->plugin, $config);
    parent::submitConfigurationForm($form, $form_state);
  }

}
