<?php

namespace Drupal\feeds_enhanced;

use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Interface SftpClientInterface
 *  Defines a class for interacting with a SFTP server.
 */
interface SftpClientInterface {

  /**
   * Returns the host name of the remote SFTP server.
   *
   * @return string
   */
  public function host(): string;

  /**
   * Returns the username for authentication on the remote SFTP server.
   *
   * @return string
   */
  public function username(): string;

  /**
   * Returns the password for authentication on the remote SFTP server.
   *
   * @return string
   */
  public function password(): string;

  /**
   * Tests connectivity to the remote host and returns TRUE on a successful
   * connection.
   *
   * @return bool
   *   TRUE if the connection is successful.
   */
  public function test(): bool;

  /**
   * Returns TRUE if a current authenticated session is running.
   *
   * @return bool
   */
  public function isLoggedIn(): bool;

  /**
   * Attempts to download a file from the remote SFTP server.
   *
   * @param $source
   *   The file path on the remote SFTP server.
   *
   * @param $destination
   *   The local path to save the file. The contents are saved directly to the
   *   specified file path.
   *
   * @return bool
   *   TRUE if the file was successfully downloaded.
   */
  public function getFile($source, &$destination): bool;

  /**
   * Assigns the logger to use.
   *
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *
   * @return $this
   *   Reference to self.
   */
  public function setLogger(LoggerChannelInterface $logger): static;

  /**
   * Returns a reference to the logger service.
   *
   * @return \Drupal\Core\Logger\LoggerChannelInterface
   */
  public function logger(): LoggerChannelInterface;

  /**
   * Returns the default timeout for SFTP requests, in seconds.
   *
   * @return int
   */
  public static function defaultTimeout(): int;

  /**
   * Returns the default port for SFTP connections.
   *
   * @return int
   */
  public static function defaultPort(): int;

}
