<?php

namespace Drupal\feeds_enhanced\Feeds\Fetcher;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\feeds\Exception\EmptyFeedException;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\File\FeedsFileSystemInterface;
use Drupal\feeds\Plugin\Type\Fetcher\FetcherInterface;
use Drupal\feeds\Plugin\Type\PluginBase;
use Drupal\feeds\Result\FetcherResult;
use Drupal\feeds\StateInterface;
use Drupal\feeds_enhanced\Result\SftpFetcherResult;
use Drupal\feeds_enhanced\SftpClient;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SftpFetcher.
 *
 * @FeedsFetcher(
 *    id = "sftp",
 *    title = @Translation("Download via SFTP"),
 *    description = @Translation("Downloads data from a secure SFTP source using the provided credentials."),
 *    form = {
 *      "configuration" = "Drupal\feeds_enhanced\Feeds\Fetcher\Form\SftpFetcherForm",
 *      "feed" = "Drupal\feeds_enhanced\Feeds\Fetcher\Form\SftpFetcherFeedForm",
 *    }
 *  )
 *
 * @package Drupal\feeds_enhanced\Feeds\Fetcher
 */
class SftpFetcher extends PluginBase implements
  FetcherInterface,
  ContainerFactoryPluginInterface {

  /**
   * Constructs an UploadFetcher object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin id.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel for this plugin.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The Drupal file system helper.
   * @param \Drupal\feeds\File\FeedsFileSystemInterface $feedsFileSystem
   *   The Drupal file system helper for Feeds.
   * @param \Drupal\key\KeyRepositoryInterface $keyRepository
   *   The key repository service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    protected LoggerChannelInterface $logger,
    protected FileSystemInterface $fileSystem,
    protected FeedsFileSystemInterface $feedsFileSystem,
    protected KeyRepositoryInterface $keyRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('feeds_enhanced.logger'),
      $container->get('file_system'),
      $container->get('feeds.file_system.in_progress'),
      $container->get('key.repository'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    return [
      'host' => '',
      'port' => SftpClient::defaultPort(),
      'username' => '',
      'password' => '',
      'timeout' => SftpClient::defaultTimeout(),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function defaultFeedConfiguration(): array {
    return $this->defaultConfiguration();
  }

  /**
   * {@inheritDoc}
   */
  public function fetch(
    FeedInterface $feed,
    StateInterface $state,
  ): FetcherResult {
    $config = [
      ...$feed->getConfigurationFor($this),
      'password' => $this->getPw($feed),
    ];

    // Parse host:port after token expansion.
    $host = $config['host'] ?? '';
    if (str_contains($host, ':')) {
      $parsed = parse_url("sftp://{$host}");
      if (isset($parsed['host'])) {
        $config['host'] = $parsed['host'];
      }
      if (isset($parsed['port'])) {
        $config['port'] = $parsed['port'];
      }
    }

    $sftp = new SftpClient($config);
    $sftp->setLogger($this->logger);
    $sink = $this->feedsFileSystem->tempnam($feed, 'sftp_fetcher_');
    if (!$sftp->getFile($feed->getSource(), $sink)) {
      $state->setMessage($this->t('The feed has not been updated.'));
      throw new EmptyFeedException();
    }
    return new SftpFetcherResult($sink, $this->fileSystem);
  }

  /**
   * Returns the password from the Key Repository.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity.
   *
   * @return string
   *   The decrypted password.
   */
  protected function getPw(FeedInterface $feed): string {
    $config = $feed->getConfigurationFor($this);
    $password_key = $config['password'] ?? '';
    $key = $this->keyRepository->getKey($password_key);
    return $key ? $key->getKeyValue() : '';
  }

}
