<?php

namespace Drupal\feeds_enhanced\Feeds\Target;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Trait FeedTargetSupportsMultiValueTrait.
 *  Trait that is intended to be paired with and to fulfill the
 *  Drupal\feeds_enhanced\Feeds\Target\FeedTargetSupportsMultiValueInterface
 *  interface's contract.
 */
trait FeedTargetSupportsMultiValueTrait {
  use StringTranslationTrait;

  /**
   * {@inheritDoc}
   */
  public function supportsMultiValue(): bool {
    $config = $this->getConfiguration();
    return array_key_exists(static::multiValuePropertyName(), $config)
      && $config[static::multiValuePropertyName()] !== FALSE
      && $config[static::multiValuePropertyName()] !== 1;
  }

  /**
   * {@inheritDoc}
   */
  public static function multiValuePropertyName(): string {
    return 'collect_multi_values';
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    return [
      ...parent::defaultConfiguration(),
      static::multiValuePropertyName() => FALSE,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $cardinality = $this->getCardinality();
    if ($cardinality === 1) {
      return parent::buildConfigurationForm($form, $form_state);
    }
    return [
      static::multiValuePropertyName() => [
        '#type' => 'checkbox',
        '#title' => $this->t('Collect multiple values'),
        '#description' => $this->t('Fields that accept multiple values will have subsequent values appended to the field instead of replacing the existing field value.'),
        '#default_value' => $this->supportsMultiValue(),
        '#return_value' => $cardinality,
      ],
      ...parent::buildConfigurationForm($form, $form_state),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary(): array {
    if ($this->getCardinality() === 1) {
      return parent::getSummary();
    }
    return [
      $this->t('Collect multiple values: @value', [
        '@value' => $this->supportsMultiValue()
          ? $this->t('Yes')
          : $this->t('No'),
      ]),
      ...parent::getSummary(),
    ];
  }

  /**
   * Returns the cardinality of this target's field.
   *
   * @return int
   *   The cardinality of the target field.
   */
  protected function getCardinality(): int {
    return $this->targetDefinition?->getFieldDefinition()
      ?->getFieldStorageDefinition()
      ?->getCardinality() ?? 1;
  }

}
