<?php

namespace Drupal\feeds_enhanced;

use Drupal\Core\Logger\LoggerChannelInterface;
use phpseclib3\Net\SFTP;

/**
 * Class SftpClient
 *  Implementation of a SFTP client.
 *
 * @package Drupal\feeds_enhanced
 */
class SftpClient implements SftpClientInterface {

  /**
   * Stores the remote host name.
   *
   * @var string
   */
  protected string $host;

  /**
   * Stores the username for authentication.
   *
   * @var string
   */
  protected string $username;

  /**
   * Stores the password for authentication.
   *
   * @var string
   */
  protected string $password;

  /**
   * Reference to a SFTP client.
   *
   * @var \phpseclib3\Net\SFTP
   */
  protected SFTP $client;

  /**
   * Logger channel.
   */
  protected ?LoggerChannelInterface $logger = NULL;

  /**
   * SftpClient constructor.
   *
   * @param array $data
   *   Connection configuration data.
   * @param \Drupal\Core\Logger\LoggerChannelInterface|null $logger
   *   Logger channel (optional).
   */
  public function __construct(array $data, ?LoggerChannelInterface $logger = NULL) {
    [
      'host' => $this->host,
      'username' => $this->username,
      'password' => $this->password,
    ] = $data;
    $port = $data['port'] ?? static::defaultPort();
    $this->client = new SFTP($this->host, $port);
    if ($timeout = $data['timeout'] ?? NULL) {
      $this->client->setTimeout($timeout);
    }
    $this->logger = $logger;
  }

  /**
   * {@inheritDoc}
   */
  public function host(): string {
    return $this->host;
  }

  /**
   * {@inheritDoc}
   */
  public function username(): string {
    return $this->username;
  }

  /**
   * {@inheritDoc}
   */
  public function password(): string {
    return $this->password;
  }

  /**
   * {@inheritDoc}
   */
  public function test(): bool {
    return $this->logIn();
  }

  /**
   * {@inheritDoc}
   */
  public function isLoggedIn(): bool {
    return $this->client->isConnected() && $this->client->isAuthenticated();
  }

  /**
   * {@inheritDoc}
   */
  public function getFile($source, &$destination): bool {
    if (!$this->logIn()) {
      $this
        ->logger()
        ->warning('Failed to log in to the SFTP server at @host.', [
          '@host' => $this->host,
        ]);
      return FALSE;
    }
    if (!$result = $this->client->get($source, $destination)) {
      $this
        ->logger()
        ->warning('Failed to get the file @source from the SFTP server at @host.', [
          '@source' => $source,
          '@host' => $this->host,
        ]);
    }
    return $result;
  }

  /**
   * Attempts to log in to the SFTP server.
   *
   * @return bool
   *   Returns TRUE on success.
   */
  public function logIn(): bool {
    if ($this->isLoggedIn()) {
      return TRUE;
    }
    return $this->client->login($this->username, $this->password);
  }

  /**
   * {@inheritDoc}
   */
  public function setLogger(LoggerChannelInterface $logger): static {
    $this->logger = $logger;
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function logger(): LoggerChannelInterface {
    if ($this->logger === NULL) {
      throw new \LogicException('Logger was not provided to SftpClient.');
    }
    return $this->logger;
  }

  /**
   * {@inheritDoc}
   */
  public static function defaultTimeout(): int {
    return 90;
  }

  /**
   * Returns the default SFTP port.
   *
   * @return int
   *   The default SFTP port (22).
   */
  public static function defaultPort(): int {
    return 22;
  }

}
