<?php

namespace Drupal\Tests\feeds_enhanced\Functional;

use Drupal\feeds_enhanced\SftpClient;

/**
 * Test-specific SFTP client that extends the real SftpClient.
 *
 * This class allows us to test the Key integration and SFTP functionality
 * without making actual network connections.
 */
class TestSftpClient extends SftpClient {

  /**
   * The path to the test file to return instead of making SFTP calls.
   *
   * @var string
   */
  protected string $testFilePath;

  /**
   * Overrides the parent constructor to accept a test file path.
   *
   * @param array $data
   *   The configuration data.
   * @param string $testFilePath
   *   The path to the test file to use instead of SFTP.
   */
  public function __construct(array $data, string $testFilePath) {
    $this->testFilePath = $testFilePath;
    // Store the config but don't call parent constructor to avoid SFTP connection.
    $this->host = $data['host'] ?? '';
    $this->username = $data['username'] ?? '';
    $this->password = $data['password'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function test(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isLoggedIn(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function logIn(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($source, &$destination): bool {
    // Copy our test file to the destination instead of using SFTP.
    $content = file_get_contents($this->testFilePath);
    if ($content === FALSE) {
      $this->logger()->error('Failed to read test file: @file', ['@file' => $this->testFilePath]);
      return FALSE;
    }

    $result = file_put_contents($destination, $content) !== FALSE;
    $this->logger()->info('TestSftpClient::getFile result: @result', ['@result' => $result ? 'SUCCESS' : 'FAILED']);

    return $result;
  }

  /**
   * Returns the password that was decrypted from the Key.
   *
   * This allows us to verify that the Key integration is working.
   *
   * @return string
   */
  public function getDecryptedPassword(): string {
    return $this->password;
  }

}
