<?php

namespace Drupal\feeds_enhanced\Feeds\Parser;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Feeds\Item\DynamicItem;
use Drupal\feeds\Feeds\Parser\ParserBase;
use Drupal\feeds\Result\FetcherResultInterface;
use Drupal\feeds\Result\ParserResult;
use Drupal\feeds\Result\ParserResultInterface;
use Drupal\feeds\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class EntityDataParser
 *  Returns entity field data as parsed data, based on the Feed type's defined
 *  entity type ID and bundle name. The data returned are defined by this Feed
 *  type's mapping.
 *
 *  This parser ignores the fetcher result completely. For best results,
 *  use the Null Data Source fetcher in conjunction with this parser.
 *
 * @package Drupal\feeds_enhanced\Feeds\Parser
 *
 * @FeedsParser(
 *   id = "entity_data",
 *   title = @Translation("Entity data"),
 *   description = @Translation("Returns existing entity data from the source fields specified under Mappings. Does not read from fetcher input. For optimal results, use the <strong>Null data source</strong> fetcher.")
 * )
 */
class EntityDataParser extends ParserBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function parse(
    FeedInterface $feed,
    FetcherResultInterface $fetcher_result,
    StateInterface $state,
  ): ParserResultInterface {
    $result = new ParserResult();
    $entities = $this->getTargetEntities();
    foreach ($entities as $entity) {
      $item = new DynamicItem();
      foreach ($this->feedType->getMappings() as $mapping_info) {
        $field_name = $mapping_info['target'];
        if (!$entity->hasField($field_name) || !$entity->get($field_name)) {
          continue;
        }
        $item->set($field_name, array_column(
          $entity->get($field_name)->getValue(),
          'value'
        ));
      }
      $result->addItem($item);
    }

    // Report per-batch progress.
    $state->total = count($entities);
    $state->pointer = $result->count();
    $state->progress($state->total, $state->pointer);
    // Set progress to complete if no more results are parsed. Can happen with
    // empty lines in CSV.
    if (!$result->count()) {
      $state->setCompleted();
    }
    return $result;
  }

  /**
   * {@inheritDoc}
   */
  public function getMappingSources(): array {
    // Only use user-specified mappings.
    return [];
  }

  /**
   * Returns the target entity type ID for this Feed type's parser.
   *
   * @return string
   */
  protected function getTargetEntityTypeId(): string {
    return $this->feedType->getProcessor()->entityType();
  }

  /**
   * Returns the target bundle name for this Feed type's parser.
   *
   * @return string
   */
  protected function getTargetBundleName(): string {
    return $this->feedType->getProcessor()->bundle();
  }

  /**
   * Returns the key to use for current entity type's bundle/type.
   *
   * @return string
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getTargetBundleKey(): string {
    $definition = $this->entityTypeManager
      ->getDefinition($this->getTargetEntityTypeId());
    return $definition->getKey('bundle');
  }

  /**
   * Returns all target entities, as defined by this Feed type's entity type
   * and bundle processor settings.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface[]
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getTargetEntities(): array {
    $storage = $this->entityTypeManager
      ->getStorage($this->getTargetEntityTypeId());

    if ($bundle_key = $this->getTargetBundleKey()) {
      return $storage
        ->loadByProperties([$bundle_key => $this->getTargetBundleName()]) ?: [];
    }
    return $storage->loadMultiple() ?: [];
  }

}
