<?php

namespace Drupal\feeds_enhanced\Feeds\Processor;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\Exception\EmptyFeedException;
use Drupal\feeds\Exception\ValidationException;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Feeds\Item\ItemInterface;
use Drupal\feeds\Feeds\Processor\GenericContentEntityProcessor;
use Drupal\feeds\StateInterface;
use Drupal\feeds\StateType;

/**
 * Class EnhancedContentEntityProcessor
 *  Defines a processor that supports collecting field values from multiple
 *  rows into a single entity's field.
 *
 * @see \Drupal\feeds_enhanced\Hookmanager::alterProcessorPluginInfo()
 */
class EnhancedContentEntityProcessor extends GenericContentEntityProcessor {

  /**
   * Persist the entity hash between method calls.
   *
   * @var string
   */
  protected string $entityHash;

  /**
   * Persists the existing entity type ID between method calls.
   *
   * @var string|null
   */
  protected ?string $existingEntityTypeId;

  /**
   * Persist a reference to the entity being targeted by this plugin.
   *
   * @var \Drupal\Core\Entity\EntityInterface|null
   */
  protected ?EntityInterface $entity = NULL;

  /**
   * An array of field mapping data, keyed by field name.
   *
   * @var array
   */
  protected array $targets;

  /**
   * Assigns field target information from the provided feed type mappings.
   *
   * @param array $mappings
   *   An array of field mappings, keyed by field name.
   *
   * @return $this
   *   Reference to self.
   */
  protected function setTargets(array $mappings): static {
    $this->targets = [];
    foreach ($mappings as $mapping) {
      $target = $mapping['target'];
      unset($mapping['target']);
      $this->targets[$target] = $mapping;
    }
    return $this;
  }

  /**
   * Returns an array of field mapping data, keyed by field name.
   *
   * @return array
   *   An array of field mapping data, keyed by field name.
   */
  protected function targets(): array {
    return $this->targets ?? [];
  }

  /**
   * Returns the cardinality for the specified field.
   *
   * @param string $field_name
   *   The name of the field whose cardinality should be returned.
   *
   * @return int
   *   The number of values supported by this field.
   */
  protected function getCardinality(string $field_name): int {
    $mapping = $this->targets()[$field_name];
    return $mapping['settings']['collect_multi_values'] ?? 1;
  }

  /**
   * Returns TRUE if the specified field supports a multi-value mapping.
   *
   * @param string $field_name
   *   The name of the field to check.
   *
   * @return bool
   *   TRUE if the specified field supports multi-value mapping.
   */
  protected function supportsMultiValue(string $field_name): bool {
    return $this->getCardinality($field_name) !== 0
      && $this->getCardinality($field_name) !== 1;
  }

  /**
   * Returns TRUE if the specified field on the current entity has room to
   * store additional values, relative to the field's cardinality.
   *
   * @param string $field_name
   *   The name of the field on the entity.
   *
   * @return bool
   *   TRUE if the field has room for additional values.
   */
  protected function fieldHasRoom(string $field_name): bool {
    if (!$field = $this->entity->get($field_name) ?? NULL) {
      return FALSE;
    }
    $cardinality = $this->getCardinality($field_name);
    return $field->count() < $this->getCardinality($field_name)
      || $cardinality === FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED;
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    // Inject default values for enhanced entity processing.
    return [
      ...parent::defaultConfiguration(),
      'collect_multi_values' => FALSE,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function process(
    FeedInterface $feed,
    ItemInterface $item,
    StateInterface $state,
  ): void {
    /**
     * In order to support accumulating values in multi-value fields, the
     * original process method has been split up into ::processBefore() and
     * ::processAfter(). The code between these two methods and after
     * ::processAfter() provides the multi-value collection support.
     */
    // Perform initial processing and allow $entity class property to be set.
    $this->processBefore($feed, $item, $state);
    // Assign mapping information relative to multi value collection, then
    // collect the original values from the returned entity.
    $this->setTargets($feed->getType()->getMappings());
    $original_values = [];
    foreach ($this->targets() as $field_name => $mapping) {
      if (!$this->supportsMultiValue($field_name)) {
        continue;
      }
      $original_values[$field_name] = $this->entity->get($field_name)
        ->getValue();
    }

    // Allow the processor to complete, then combine the original values with
    // the newly added value(s) and assign it all back to the entity.
    $this->processAfter($feed, $item, $state);
    foreach ($this->targets() as $field_name => $mapping) {
      if (!$this->supportsMultiValue($field_name)) {
        continue;
      }

      if (!array_key_exists($field_name, $original_values)) {
        continue;
      }
      if (!$field = $this->entity->get($field_name) ?? NULL) {
        continue;
      }
      $property = key($mapping['map']);
      $original_values = array_column($original_values[$field_name], $property);
      $term_values = array_column($field->getValue() ?? [], $property);
      $diff = array_diff($original_values, $term_values);
      if (empty($diff)) {
        continue;
      }
      foreach ($diff as $value) {
        if (!$this->fieldHasRoom($field_name)) {
          break;
        }
        $field->appendItem([$property => $value]);
      }
    }
    $this->entity->save();
  }

  //
  // --- Original code below --- //
  //

  /**
   * Perform the first part of the feed import process routine.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed to process.
   *
   * @param \Drupal\feeds\Feeds\Item\ItemInterface $item
   *   The item to process.
   *
   * @param \Drupal\feeds\StateInterface $state
   *   The state of the import process.
   *
   * @return void
   */
  protected function processBefore(
    FeedInterface $feed,
    ItemInterface $item,
    StateInterface $state,
  ): void {
    // Initialize clean list if needed.
    /** @var \Drupal\feeds\StateInterface $clean_state */
    $clean_state = $feed->getState(StateInterface::CLEAN);
    if (!$clean_state->initiated()) {
      $this->initCleanList($feed, $clean_state);
    }
    [
      'insert_new' => $insert_new,
      'update_existing' => $update_existing,
    ] = $this->configuration;
    $skip_new = $insert_new == static::SKIP_NEW;
    $this->existingEntityTypeId = $this->existingEntityId($feed, $item);
    $skip_existing = $update_existing == static::SKIP_EXISTING;
    // If the entity is an existing entity it must be removed from the clean
    // list.
    if ($this->existingEntityTypeId) {
      $clean_state->removeItem($this->existingEntityTypeId);
    }
    // Skip existing entities when configured to do so.
    if ($skip_existing && $this->existingEntityTypeId) {
      $state->report(StateType::SKIP, 'Skipped because the entity already exists.', [
        'feed' => $feed,
        'item' => $item,
        'entity_label' => [$this->existingEntityTypeId, 'id'],
      ]);
      return;
    }
    // Skip new entities when configured to do so.
    if (!$this->existingEntityTypeId && $skip_new) {
      $state->report(StateType::SKIP, 'Skipped because creating new entities is disabled.', [
        'feed' => $feed,
        'item' => $item,
      // No entity ID exists for new entities.
        'entity_label' => 'N/A',
      ]);
      return;
    }

    // Delay building a new entity until necessary.
    if ($this->existingEntityTypeId) {
      $this->entity = $this->storageController->load($this->existingEntityTypeId);
    }
    $this->entityHash = $this->hash($item);
    if (isset($this->entity) && $this->entity) {
      $item_hash = $this->entity->get('feeds_item')->getItemHashByFeed($feed);
    }
    $changed = $this->existingEntityTypeId && $this->entityHash !== $item_hash;

    // Do not proceed if the item exists, has not changed, and we're not
    // forcing the update.
    if (
      $this->existingEntityTypeId
        && !$changed
        && !($this->configuration['skip_hash_check'] ?? NULL)
    ) {
      $state
        ->report(StateType::SKIP, 'Skipped because the source data has not changed.', [
          'feed' => $feed,
          'item' => $item,
          'entity' => $this->entity,
          'entity_label' => $this->identifyEntity($this->entity, $feed),
        ]);
      return;
    }
    // Build a new entity.
    if (!$this->existingEntityTypeId && !$skip_new) {
      $this->entity = $this->newEntity($feed);
    }
  }

  /**
   * Perform the last part of the fed import process routine.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed to process.
   *
   * @param \Drupal\feeds\Feeds\Item\ItemInterface $item
   *   The item to process.
   *
   * @param \Drupal\feeds\StateInterface $state
   *   The state of the import process.
   *
   * @return void
   */
  protected function processAfter(
    FeedInterface $feed,
    ItemInterface $item,
    StateInterface $state,
  ): void {
    try {
      // Set feeds_item values.
      $feeds_item = $this->entity->get('feeds_item')->getItemByFeed($feed, TRUE);
      $feeds_item->hash = $this->entityHash;
      // Set new revision if needed.
      if ($this->configuration['revision'] ?? NULL) {
        $this->entity->setNewRevision(TRUE);
        $this->entity->setRevisionCreationTime($this->dateTime->getRequestTime());
      }
      // Set field values.
      $this->map($feed, $this->entity, $item);
      // Validate the entity.
      $feed->dispatchEntityEvent(
        FeedsEvents::PROCESS_ENTITY_PREVALIDATE,
        $this->entity,
        $item
      );
      $this->entityValidate($this->entity, $feed);
      // Dispatch pre-save event.
      $feed->dispatchEntityEvent(
        FeedsEvents::PROCESS_ENTITY_PRESAVE,
        $this->entity,
        $item
      );
      // This will throw an exception on failure.
      $this->entitySaveAccess($this->entity);
      // Set imported time.
      $feeds_item->imported = $this->dateTime->getRequestTime();
      // And... Save! We made it.
      $this->entity->save();
      // Dispatch post-save event.
      $feed->dispatchEntityEvent(
        FeedsEvents::PROCESS_ENTITY_POSTSAVE,
        $this->entity,
        $item
      );
      // Track progress.
      $operation = $this->existingEntityTypeId
        ? StateType::UPDATE
        : StateType::CREATE;
      $state->report($operation, '', [
        'feed' => $feed,
        'item' => $item,
        'entity' => $this->entity,
        'entity_label' => $this->identifyEntity($this->entity, $feed),
      ]);
    }
    catch (EmptyFeedException $e) {
      // Not an error.
      $state
        ->report(StateType::SKIP, 'Skipped because a value appeared to be empty.', [
          'feed' => $feed,
          'item' => $item,
          'entity' => $this->entity,
          'entity_label' => $this->identifyEntity($this->entity, $feed),
        ]);
    }
    // Something bad happened. Log it.
    catch (ValidationException $e) {
      $state->report(StateType::FAIL, $e->getFormattedMessage(), [
        'feed' => $feed,
        'item' => $item,
        'entity' => $this->entity,
        'entity_label' => $this->identifyEntity($this->entity, $feed),
      ]);
      $state->setMessage($e->getFormattedMessage(), 'warning');
    }
    catch (\Exception $e) {
      $state->report(StateType::FAIL, $e->getMessage(), [
        'feed' => $feed,
        'item' => $item,
        'entity' => $this->entity,
        'entity_label' => $this->identifyEntity($this->entity, $feed),
      ]);
      $state->setMessage($e->getMessage(), 'warning');
    }
  }

}
