<?php

namespace Drupal\Tests\feeds_enhanced\Functional;

use Drupal\feeds\Entity\Feed;
use Drupal\feeds\Entity\FeedType;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests SFTP Fetcher host:port configuration handling.
 *
 * @group feeds_enhanced
 */
class SftpFetcherFormTest extends BrowserTestBase
{

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced',
    'key',
    'node',
    'user',
    'text',
    'filter',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();
  }

  /**
   * Tests new feed inherits feed type config.
   */
  public function testNewFeedInheritsFeedTypeConfig(): void
  {
    $feed_type = FeedType::create([
      'id' => 'sftp_test',
      'label' => 'SFTP Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'sftp.example.com',
        'port' => 2222,
        'username' => 'testuser',
        'password' => '',
        'timeout' => 90,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'sftp_test',
      'title' => 'Test Feed',
    ]);

    $fetcher = $feed_type->getFetcher();
    $form_object = \Drupal::classResolver()->getInstanceFromDefinition(
      $fetcher->getPluginDefinition()['form']['feed'],
    );
    $form_object->setPlugin($fetcher);

    $form_state = new \Drupal\Core\Form\FormState();
    $form = $form_object->buildConfigurationForm([], $form_state, $feed);

    $this->assertNotEmpty($form);
  }

  /**
   * Tests configuration stores host and port separately.
   */
  public function testConfigurationStoresHostPortSeparately(): void
  {
    $feed_type = FeedType::create([
      'id' => 'sftp_test',
      'label' => 'SFTP Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'example.com',
        'port' => 3000,
        'username' => 'user123',
        'password' => '',
        'timeout' => 60,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $config = $feed_type->getFetcher()->getConfiguration();
    $this->assertEquals('example.com', $config['host']);
    $this->assertEquals(3000, $config['port']);
    $this->assertEquals('user123', $config['username']);
  }

  /**
   * Tests form trait parses host:port correctly.
   */
  public function testFormTraitParsesHostPort(): void
  {
    $feed_type = FeedType::create([
      'id' => 'sftp_test',
      'label' => 'SFTP Test',
      'fetcher' => 'sftp',
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'sftp_test',
      'title' => 'Test',
    ]);

    $fetcher = $feed_type->getFetcher();
    $form_object = \Drupal::classResolver()->getInstanceFromDefinition(
      $fetcher->getPluginDefinition()['form']['feed'],
    );
    $form_object->setPlugin($fetcher);

    $reflection = new \ReflectionClass($form_object);
    $method = $reflection->getMethod('parseHostAndPort');
    $method->setAccessible(TRUE);

    $result = $method->invoke($form_object, 'example.com:3000');
    $this->assertEquals('example.com', $result['host']);
    $this->assertEquals(3000, $result['port']);

    $result = $method->invoke($form_object, '127.0.0.1:22');
    $this->assertEquals('127.0.0.1', $result['host']);
    $this->assertEquals(22, $result['port']);
  }

}
