<?php

namespace Drupal\Tests\feeds_enhanced\Functional;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\feeds_enhanced\SftpClientInterface;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * A mock SFTP client that uses a local test file.
 */
class MockSftpClient implements SftpClientInterface {

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The path to the test file.
   *
   * @var string
   */
  protected $testFilePath;

  /**
   * Constructs a new MockSftpClient object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param string $testFilePath
   *   The path to the test file.
   */
  public function __construct(LoggerChannelInterface $logger, $testFilePath) {
    $this->logger = $logger;
    $this->testFilePath = $testFilePath;
  }

  /**
   * {@inheritdoc}
   */
  public function host(): string {
    return 'example.com';
  }

  /**
   * {@inheritdoc}
   */
  public function username(): string {
    return 'user';
  }

  /**
   * {@inheritdoc}
   */
  public function password(): string {
    return 'password';
  }

  /**
   * {@inheritdoc}
   */
  public function test(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isLoggedIn(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($source, &$destination): bool {
    // Log that the mock is being called (without echo for cleaner test output)
    $this->logger->info('MockSftpClient::getFile called with source: @source, destination: @destination', [
      '@source' => $source,
      '@destination' => $destination,
    ]);

    // Use file_get_contents and file_put_contents for robustness.
    $content = file_get_contents($this->testFilePath);
    if ($content === FALSE) {
      $this->logger->error('Failed to read test file: @file', ['@file' => $this->testFilePath]);
      return FALSE;
    }

    $result = file_put_contents($destination, $content) !== FALSE;
    $this->logger->info('MockSftpClient::getFile result: @result', ['@result' => $result ? 'SUCCESS' : 'FAILED']);

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function setLogger(LoggerChannelInterface $logger): static {
    $this->logger = $logger;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function logger(): LoggerChannelInterface {
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultTimeout(): int {
    return 90;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultPort(): int {
    return 22;
  }

  /**
   * {@inheritdoc}
   */
  public function logIn(): bool {
    return TRUE;
  }

}

/**
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
 * @group feeds_enhanced
 */
class SftpFetcherFunctionalTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced',
    'node',
    'user',
    'text',
    'filter',
  ];

  /**
   * The test CSV file path.
   *
   * @var string
   */
  protected $csvFile;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create test CSV file with content.
    $this->csvFile = $this->createTestFile();
    $this->assertFileExists($this->csvFile);

    // Create our mock SFTP client service.
    $this->setUpMockSftpClient();
  }

  /**
   * Creates a test CSV file in the public files directory.
   *
   * @return string
   *   The path to the created file.
   */
  protected function createTestFile() {
    $file_path = 'public://test_feed.csv';
    $csv_content = "title,body\nTest Article 1,This is the body of test article 1\nTest Article 2,This is the body of test article 2";
    file_put_contents($file_path, $csv_content);
    return $this->container->get('file_system')->realpath($file_path);
  }

  /**
   * Sets up a mock SFTP client service to bypass actual SFTP connections.
   */
  protected function setUpMockSftpClient() {
    $mockSftpClient = new MockSftpClient(
      $this->container->get('logger.factory')->get('feeds_enhanced'),
      $this->csvFile
    );

    // Override the service with our mock - use the kernel container.
    $this->container->get('kernel')->getContainer()->set('feeds_enhanced.sftp_client', $mockSftpClient);

    // Also set it in the current container.
    $this->container->set('feeds_enhanced.sftp_client', $mockSftpClient);
  }

  /**
   * Tests the SFTP service registration and basic functionality.
   */
  public function testSftpServiceRegistration() {
    // Test that our mock service is properly registered.
    $this->assertTrue(\Drupal::hasService('feeds_enhanced.sftp_client'), 'SFTP client service should be available');

    $service = \Drupal::service('feeds_enhanced.sftp_client');
    $this->assertInstanceOf(MockSftpClient::class, $service);

    // Test basic service functionality.
    $this->assertEquals('example.com', $service->host());
    $this->assertEquals('user', $service->username());
    $this->assertTrue($service->test());
    $this->assertTrue($service->isLoggedIn());

    // Test file retrieval through the mock.
    $tempFile = tempnam(sys_get_temp_dir(), 'sftp_test');
    $result = $service->getFile('test.csv', $tempFile);
    $this->assertTrue($result, 'Mock SFTP service should successfully retrieve file');

    // Verify the content.
    $content = file_get_contents($tempFile);
    $this->assertStringContainsString('Test Article 1', $content);
    $this->assertStringContainsString('Test Article 2', $content);

    // Test CSV structure.
    $csv_lines = explode("\n", trim($content));
    $this->assertCount(3, $csv_lines, 'CSV should have header + 2 data rows');
    $this->assertEquals('title,body', $csv_lines[0], 'First line should be CSV header');

    unlink($tempFile);
  }

}
