<?php

namespace Drupal\Tests\feeds_enhanced\Kernel;

use Drupal\feeds\Entity\Feed;
use Drupal\feeds\Entity\FeedType;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Tests SFTP Fetcher with token expansion.
 *
 * Validates that tokens in the host field are expanded at runtime
 * and correctly parsed into separate host and port values.
 *
 * @group feeds_enhanced
 */
class SftpFetcherTokenExpansionTest extends KernelTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced',
    'feeds_enhanced_tokens',
    'pantheon_si_tokens',
    'key',
    'token',
    'node',
    'user',
    'system',
    'field',
    'text',
    'filter',
    'file',
    'options',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('feeds_feed');
    $this->installEntitySchema('key');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig([
      'feeds',
      'node',
      'filter',
      'pantheon_si_tokens',
    ]);

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Define a test constant that simulates Pantheon SI tunnel.
    if (!defined('PANTHEON_SOIP_TEST_PORT')) {
      define('PANTHEON_SOIP_TEST_PORT', '40123');
    }

    // Register the test constant in pantheon_si_tokens config.
    // Config is a simple array of constant names.
    $config = $this->config('pantheon_si_tokens.settings');
    $config->set('constants', ['PANTHEON_SOIP_TEST_PORT']);
    $config->save();

    // Clear all caches to ensure tokens are available.
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['token_info']);
    \Drupal::service('kernel')->invalidateContainer();
    \Drupal::service('cache.discovery')->deleteAll();
  }

  /**
   * Tests token expansion in SFTP host field during fetch.
   */
  public function testTokenExpansionInHostField(): void
  {
    // Create a feed type with token in host field.
    // Token name is constant name in lowercase.
    $feed_type = FeedType::create([
      'id' => 'sftp_token_test',
      'label' => 'SFTP Token Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => '127.0.0.1:[pantheon_si_tunnel:pantheon_soip_test_port]',
        'username' => 'testuser',
        'password' => '',
        'timeout' => 30,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    // Create a feed.
    $feed = Feed::create([
      'type' => 'sftp_token_test',
      'title' => 'Test Feed',
      'source' => '/test/path/file.csv',
    ]);
    $feed->save();

    // Get the fetcher plugin.
    $fetcher = $feed_type->getFetcher();

    // Verify constant is defined.
    $this->assertTrue(
      defined('PANTHEON_SOIP_TEST_PORT'),
      'Test constant should be defined'
    );
    $this->assertEquals('40123', PANTHEON_SOIP_TEST_PORT);

    // Verify config has the constant registered.
    $registered_constants = $this->config('pantheon_si_tokens.settings')->get('constants');
    $this->assertContains('PANTHEON_SOIP_TEST_PORT', $registered_constants);

    // Test direct token replacement to verify token works.
    $test_replacement = \Drupal::token()->replace(
      '[pantheon_si_tunnel:pantheon_soip_test_port]',
      [],
      ['clear' => FALSE]
    );
    $this->assertEquals('40123', $test_replacement, 'Direct token replacement should work');

    // Manually expand tokens (simulating what feeds_enhanced_tokens does).
    $token_service = \Drupal::token();
    $config = $feed->getConfigurationFor($fetcher);

    // If feed doesn't have overrides, use feed type config.
    if (empty($config['host'])) {
      $config = $feed_type->getFetcher()->getConfiguration();
    }

    // Debug: Check if we have a host value.
    $this->assertNotEmpty($config, 'Feed config should not be empty');
    $this->assertArrayHasKey('host', $config, 'Feed config should have host key');

    // Expand the host field.
    $expanded_host = $token_service->replace(
      $config['host'],
      ['feed' => $feed],
      ['clear' => TRUE]
    );

    // Verify token was expanded.
    $this->assertEquals(
      '127.0.0.1:40123',
      $expanded_host,
      'Token [pantheon_si_tunnel:pantheon_soip_test_port] should expand to 40123'
    );

    // Simulate what happens in SftpFetcher::fetch() - parse the expanded host.
    $final_host = $expanded_host;
    $final_port = 22;
    if (str_contains($expanded_host, ':')) {
      $parsed = parse_url("sftp://{$expanded_host}");
      $final_host = $parsed['host'] ?? $expanded_host;
      $final_port = $parsed['port'] ?? 22;
    }

    // Verify parsing extracted correct values.
    $this->assertEquals('127.0.0.1', $final_host, 'Host should be extracted correctly');
    $this->assertEquals(40123, $final_port, 'Port should be extracted from expanded token');
  }

  /**
   * Tests token expansion with hostname and port token.
   */
  public function testTokenExpansionWithHostnameAndPort(): void
  {
    $feed_type = FeedType::create([
      'id' => 'sftp_hostname_test',
      'label' => 'SFTP Hostname Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'sftp.example.com:[pantheon_si_tunnel:pantheon_soip_test_port]',
        'username' => 'testuser',
        'password' => '',
        'timeout' => 30,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'sftp_hostname_test',
      'title' => 'Hostname Test Feed',
      'source' => '/test/file.csv',
    ]);
    $feed->save();

    $fetcher = $feed_type->getFetcher();
    $config = $feed->getConfigurationFor($fetcher);

    // If feed doesn't have overrides, use feed type config.
    if (empty($config['host'])) {
      $config = $feed_type->getFetcher()->getConfiguration();
    }

    // Expand tokens.
    $token_service = \Drupal::token();
    $expanded_host = $token_service->replace(
      $config['host'],
      ['feed' => $feed],
      ['clear' => TRUE]
    );

    $this->assertEquals(
      'sftp.example.com:40123',
      $expanded_host,
      'Hostname with port token should expand correctly'
    );

    // Parse expanded value.
    $final_host = $expanded_host;
    $final_port = 22;
    if (str_contains($expanded_host, ':')) {
      $parsed = parse_url("sftp://{$expanded_host}");
      $final_host = $parsed['host'] ?? $expanded_host;
      $final_port = $parsed['port'] ?? 22;
    }

    $this->assertEquals('sftp.example.com', $final_host);
    $this->assertEquals(40123, $final_port);
  }

  /**
   * Tests that static port numbers still work.
   */
  public function testStaticPortNumber(): void
  {
    $feed_type = FeedType::create([
      'id' => 'sftp_static_test',
      'label' => 'SFTP Static Port Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'sftp.example.com:2222',
        'username' => 'testuser',
        'password' => '',
        'timeout' => 30,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'sftp_static_test',
      'title' => 'Static Port Test Feed',
      'source' => '/test/file.csv',
    ]);
    $feed->save();

    $fetcher = $feed_type->getFetcher();
    $config = $feed->getConfigurationFor($fetcher);

    // If feed doesn't have overrides, use feed type config.
    if (empty($config['host'])) {
      $config = $feed_type->getFetcher()->getConfiguration();
    }

    $host = $config['host'];

    // Parse the static port.
    $final_host = $host;
    $final_port = 22;
    if (str_contains($host, ':')) {
      $parsed = parse_url("sftp://{$host}");
      $final_host = $parsed['host'] ?? $host;
      $final_port = $parsed['port'] ?? 22;
    }

    $this->assertEquals('sftp.example.com', $final_host);
    $this->assertEquals(2222, $final_port, 'Static port should be parsed correctly');
  }

}
