<?php

namespace Drupal\Tests\feeds_enhanced\Unit;

use Drupal\feeds_enhanced\Feeds\Fetcher\Form\SftpFetcherFormTrait;
use Drupal\feeds_enhanced\SftpClient;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the SftpFetcherFormTrait.
 *
 * @group feeds_enhanced
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\Form\SftpFetcherFormTrait
 */
class SftpFetcherFormTraitTest extends UnitTestCase
{

  /**
   * Test class that uses the trait.
   *
   * @var object
   */
  protected $traitObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    $this->traitObject = new class {
      use SftpFetcherFormTrait {
        getHostWithPort as public;
        parseHostAndPort as public;
      }

      protected $config = [];

      protected function getConfiguration(string $config_name): mixed
      {
        return $this->config[$config_name] ?? NULL;
      }

      protected function keyRepository(): KeyRepositoryInterface
      {
        return $this->createMock(KeyRepositoryInterface::class);
      }

      public function setConfig(array $config): void
      {
        $this->config = $config;
      }

      protected function t($string, array $args = [], array $options = [])
      {
        return strtr($string, $args);
      }

    };
  }

  /**
   * Tests parseHostAndPort with hostname:port format.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortWithPort(): void
  {
    $result = $this->traitObject->parseHostAndPort('example.com:2222');

    $this->assertEquals('example.com', $result['host']);
    $this->assertEquals(2222, $result['port']);
  }

  /**
   * Tests parseHostAndPort with hostname only.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortWithoutPort(): void
  {
    $result = $this->traitObject->parseHostAndPort('example.com');

    $this->assertEquals('example.com', $result['host']);
    $this->assertEquals(SftpClient::defaultPort(), $result['port']);
  }

  /**
   * Tests parseHostAndPort with IP:port format.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortWithIp(): void
  {
    $result = $this->traitObject->parseHostAndPort('127.0.0.1:40000');

    $this->assertEquals('127.0.0.1', $result['host']);
    $this->assertEquals(40000, $result['port']);
  }

  /**
   * Tests parseHostAndPort with empty value.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortEmpty(): void
  {
    $result = $this->traitObject->parseHostAndPort('');

    $this->assertEquals('', $result['host']);
    $this->assertEquals(SftpClient::defaultPort(), $result['port']);
  }

  /**
   * Tests parseHostAndPort with token syntax before expansion.
   *
   * Token expansion happens before form submission, so if a token
   * hasn't been expanded yet, parse_url treats it as part of the path.
   * This is expected behavior.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortWithTokenBeforeExpansion(): void
  {
    $result = $this->traitObject->parseHostAndPort(
      '127.0.0.1:[pantheon_si_tunnel:port]',
    );

    $this->assertEquals(
      '127.0.0.1:[pantheon_si_tunnel:port]',
      $result['host'],
    );
    $this->assertEquals(SftpClient::defaultPort(), $result['port']);
  }

  /**
   * Tests parseHostAndPort with expanded token (numeric port).
   *
   * After token expansion, the value becomes numeric.
   *
   * @covers ::parseHostAndPort
   */
  public function testParseHostAndPortWithExpandedToken(): void
  {
    $result = $this->traitObject->parseHostAndPort('127.0.0.1:40000');

    $this->assertEquals('127.0.0.1', $result['host']);
    $this->assertEquals(40000, $result['port']);
  }

  /**
   * Tests getHostWithPort with default port.
   *
   * When port is the default (22), it's not appended to the host.
   *
   * @covers ::getHostWithPort
   */
  public function testGetHostWithPortDefault(): void
  {
    $this->traitObject->setConfig([
      'host' => 'example.com',
      'port' => SftpClient::defaultPort(),
    ]);

    $result = $this->traitObject->getHostWithPort();

    $this->assertEquals('example.com', $result);
  }

  /**
   * Tests getHostWithPort when host already contains port/token.
   *
   * @covers ::getHostWithPort
   */
  public function testGetHostWithPortWhenHostHasColon(): void
  {
    $this->traitObject->setConfig([
      'host' => '127.0.0.1:[pantheon_si_tunnel:port]',
      'port' => SftpClient::defaultPort(),
    ]);

    $result = $this->traitObject->getHostWithPort();

    $this->assertEquals('127.0.0.1:[pantheon_si_tunnel:port]', $result);
  }

  /**
   * Tests getHostWithPort with custom port.
   *
   * @covers ::getHostWithPort
   */
  public function testGetHostWithPortCustom(): void
  {
    $this->traitObject->setConfig([
      'host' => 'example.com',
      'port' => 2222,
    ]);

    $result = $this->traitObject->getHostWithPort();

    $this->assertEquals('example.com:2222', $result);
  }

  /**
   * Tests getHostWithPort with empty config.
   *
   * @covers ::getHostWithPort
   */
  public function testGetHostWithPortEmpty(): void
  {
    $this->traitObject->setConfig([]);

    $result = $this->traitObject->getHostWithPort();

    $this->assertEquals('', $result);
  }

}
