<?php

namespace Drupal\Tests\feeds_enhanced\Unit;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\feeds\FeedTypeInterface;
use Drupal\feeds\File\FeedsFileSystemInterface;
use Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher;
use Drupal\key\Entity\Key;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for SftpFetcher with Key integration.
 *
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
 * @group feeds_enhanced
 */
class SftpFetcherKeyIntegrationUnitTest extends UnitTestCase {

  /**
   * The SftpFetcher instance under test.
   *
   * @var \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
   */
  protected $sftpFetcher;

  /**
   * Mock logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * Mock file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fileSystem;

  /**
   * Mock feeds file system.
   *
   * @var \Drupal\feeds\File\FeedsFileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $feedsFileSystem;

  /**
   * Mock key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $keyRepository;

  /**
   * Mock key entity.
   *
   * @var \Drupal\key\Entity\Key|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockKey;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for all dependencies.
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->feedsFileSystem = $this->createMock(FeedsFileSystemInterface::class);
    $this->keyRepository = $this->createMock(KeyRepositoryInterface::class);
    $this->mockKey = $this->createMock(Key::class);

    // Create a mock FeedType for the configuration.
    $mockFeedType = $this->createMock(FeedTypeInterface::class);

    // Configure key repository mock.
    $this->keyRepository
      ->expects($this->any())
      ->method('getKey')
      ->with('test_key_id')
      ->willReturn($this->mockKey);

    // Configure key entity mock to return test password.
    $this->mockKey
      ->expects($this->any())
      ->method('getKeyValue')
      ->willReturn('decrypted_test_password');

    // Create SftpFetcher with mocked dependencies.
    $configuration = [
      'host' => 'example.com',
      'username' => 'testuser',
    // This is the Key ID.
      'password' => 'test_key_id',
      'timeout' => 90,
    // Required by Feeds PluginBase.
      'feed_type' => $mockFeedType,
    ];

    $this->sftpFetcher = new SftpFetcher(
      $configuration,
      'sftp',
      ['class' => SftpFetcher::class],
      $this->logger,
      $this->fileSystem,
      $this->feedsFileSystem,
      $this->keyRepository
    );
  }

  /**
   * Tests that the Key integration correctly decrypts passwords.
   *
   * @covers ::getPw
   */
  public function testGetPwDecryptsPasswordFromKey() {
    // Create a mock Feed entity.
    $mockFeed = $this->createMock(\Drupal\feeds\FeedInterface::class);
    $mockFeed->method('getConfigurationFor')->willReturn([
      'password' => 'test_key_id',
    ]);

    // Use reflection to test the protected getPw method.
    $reflection = new \ReflectionClass($this->sftpFetcher);
    $getPwMethod = $reflection->getMethod('getPw');
    $getPwMethod->setAccessible(TRUE);

    // Call the method and verify it returns the decrypted password.
    $decryptedPassword = $getPwMethod->invoke($this->sftpFetcher, $mockFeed);

    $this->assertEquals('decrypted_test_password', $decryptedPassword);
  }

  /**
   * Tests that configuration contains the Key ID before decryption.
   */
  public function testConfigurationContainsKeyId() {
    $reflection = new \ReflectionClass($this->sftpFetcher);
    $getConfigMethod = $reflection->getMethod('getConfiguration');
    $getConfigMethod->setAccessible(TRUE);

    $config = $getConfigMethod->invoke($this->sftpFetcher);

    $this->assertEquals('test_key_id', $config['password']);
    $this->assertEquals('example.com', $config['host']);
    $this->assertEquals('testuser', $config['username']);
  }

  /**
   * Tests the complete Key integration flow.
   */
  public function testKeyIntegrationFlow() {
    // Create a mock Feed entity.
    $mockFeed = $this->createMock(\Drupal\feeds\FeedInterface::class);
    $mockFeed->method('getConfigurationFor')->willReturn([
      'password' => 'test_key_id',
    ]);

    // Verify that the key repository is called with the correct Key ID.
    $this->keyRepository
      ->expects($this->once())
      ->method('getKey')
      ->with('test_key_id')
      ->willReturn($this->mockKey);

    // Verify that the key entity returns the decrypted password.
    $this->mockKey
      ->expects($this->once())
      ->method('getKeyValue')
      ->willReturn('decrypted_test_password');

    // Use reflection to test the getPw method.
    $reflection = new \ReflectionClass($this->sftpFetcher);
    $getPwMethod = $reflection->getMethod('getPw');
    $getPwMethod->setAccessible(TRUE);

    $decryptedPassword = $getPwMethod->invoke($this->sftpFetcher, $mockFeed);

    $this->assertEquals('decrypted_test_password', $decryptedPassword);
  }

  /**
   * Tests handling of missing keys.
   */
  public function testMissingKeyReturnsEmptyString() {
    // Configure key repository to return null (key not found)
    $keyRepository = $this->createMock(KeyRepositoryInterface::class);
    $keyRepository
      ->expects($this->once())
      ->method('getKey')
      ->with('nonexistent_key')
      ->willReturn(NULL);

    // Create a mock FeedType for the configuration.
    $mockFeedType = $this->createMock(FeedTypeInterface::class);

    // Create fetcher with missing key ID.
    $configuration = [
      'host' => 'example.com',
      'username' => 'testuser',
      'password' => 'nonexistent_key',
      'timeout' => 90,
    // Required by Feeds PluginBase.
      'feed_type' => $mockFeedType,
    ];

    $fetcher = new SftpFetcher(
      $configuration,
      'sftp',
      ['class' => SftpFetcher::class],
      $this->logger,
      $this->fileSystem,
      $this->feedsFileSystem,
      $keyRepository
    );

    // Create a mock Feed entity.
    $mockFeed = $this->createMock(\Drupal\feeds\FeedInterface::class);
    $mockFeed->method('getConfigurationFor')->willReturn([
      'password' => 'nonexistent_key',
    ]);

    // Use reflection to test the getPw method.
    $reflection = new \ReflectionClass($fetcher);
    $getPwMethod = $reflection->getMethod('getPw');
    $getPwMethod->setAccessible(TRUE);

    $result = $getPwMethod->invoke($fetcher, $mockFeed);

    $this->assertEquals('', $result);
  }

}
