<?php

declare(strict_types=1);

namespace Drupal\feeds_enhanced_tokens\EventSubscriber;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\Event\InitEvent;
use Drupal\feeds\Event\ImportFinishedEvent;
use Drupal\feeds_enhanced_tokens\TokenExpander;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for expanding tokens in Feeds.
 */
class TokenExpansionSubscriber implements EventSubscriberInterface
{

  /**
   * Constructs a TokenExpansionSubscriber object.
   *
   * @param \Drupal\feeds_enhanced_tokens\TokenExpander $tokenExpander
   *   The token expander service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    protected TokenExpander $tokenExpander,
    protected LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array
  {
    return [
      FeedsEvents::INIT_IMPORT => ['onInitImport', 1000],
      FeedsEvents::IMPORT_FINISHED => ['onImportFinished', -1000],
    ];
  }

  /**
   * Expands tokens before import begins.
   *
   * This method runs at INIT_IMPORT with high priority (1000) to
   * ensure tokens are expanded before any fetcher, parser, or
   * processor accesses the configuration or source values.
   *
   * Expands ALL string values in:
   * - Feed entity fields (source, label, custom fields, etc.)
   * - Feed Type plugin configurations (fetcher, parser, processor)
   *
   * @param \Drupal\feeds\Event\InitEvent $event
   *   The init event.
   */
  public function onInitImport(InitEvent $event): void
  {
    $feed = $event->getFeed();
    $feedType = $feed->getType();
    $this->expandFeedEntityFields($feed);
    $this->expandPluginConfigurations($feed, $feedType);
  }

  /**
   * Expands tokens in all Feed entity fields.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity.
   */
  protected function expandFeedEntityFields($feed): void
  {
    foreach ($feed->getFields() as $fieldItemList) {
      foreach ($fieldItemList as $fieldItem) {
        foreach ($fieldItem->getProperties() as $property) {
          $expandedValue = is_string($value = $property->getValue())
            ? $this->tokenExpander->expandString($value, $feed)
            : (is_array($value)
              ? $this->tokenExpander->expandConfiguration($value, $feed)
              : $value);
          if ($expandedValue !== $value) {
            $property->setValue($expandedValue);
          }
        }
      }
    }
  }

  /**
   * Expands tokens in all plugin configurations.
   *
   * Expands both feed type-level and feed-level plugin configurations.
   *
   * @param \Drupal\feeds\FeedInterface $feed
   *   The feed entity.
   * @param \Drupal\feeds\FeedTypeInterface $feedType
   *   The feed type entity.
   */
  protected function expandPluginConfigurations($feed, $feedType): void
  {
    $plugins = [
      $feedType->getFetcher(),
      $feedType->getParser(),
      $feedType->getProcessor(),
    ];

    foreach ($plugins as $plugin) {
      // Expand feed type-level plugin configuration.
      $config = $plugin->getConfiguration();
      $expandedConfig = $this->tokenExpander
        ->expandConfiguration($config, $feed);
      if ($expandedConfig !== $config) {
        $plugin->setConfiguration($expandedConfig);
      }
      // NOTE: We do NOT expand feed-level plugin config here because
      // calling $feed->setConfigurationFor() would persist the expanded
      // values back to the feed entity, replacing the original tokens.
      // Instead, individual plugins (like SftpFetcher) should expand
      // tokens at runtime when they read the config.
    }
  }

  /**
   * Clears token cache after import finishes.
   *
   * @param \Drupal\feeds\Event\ImportFinishedEvent $event
   *   The import finished event.
   */
  public function onImportFinished(ImportFinishedEvent $event): void
  {
    $this->tokenExpander->clearCache();
  }

}
