<?php

declare(strict_types=1);

namespace Drupal\Tests\feeds_enhanced_tokens\Kernel;

use Drupal\feeds\Event\FeedsEvents;
use Drupal\feeds\Event\InitEvent;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\FeedTypeInterface;
use Drupal\feeds_enhanced_tokens\EventSubscriber\TokenExpansionSubscriber;
use Drupal\feeds_enhanced_tokens\TokenExpander;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the TokenExpansionSubscriber.
 *
 * @group feeds_enhanced_tokens
 * @coversDefaultClass
 *   \Drupal\feeds_enhanced_tokens\EventSubscriber\TokenExpansionSubscriber
 */
class TokenExpansionSubscriberTest extends KernelTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced_tokens',
    'token',
    'user',
    'system',
    'field',
    'options',
  ];

  /**
   * The token expander service.
   *
   * @var \Drupal\feeds_enhanced_tokens\TokenExpander
   */
  protected TokenExpander $tokenExpander;

  /**
   * The event subscriber.
   *
   * @var \Drupal\feeds_enhanced_tokens\EventSubscriber\TokenExpansionSubscriber
   */
  protected TokenExpansionSubscriber $subscriber;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('feeds_feed');
    $this->installEntitySchema('feeds_feed_type');
    $this->installConfig([
      'feeds',
      'feeds_enhanced_tokens',
      'system',
    ]);

    $this->tokenExpander = $this->container->get(
      'feeds_enhanced_tokens.token_expander',
    );
    $logger = $this->container->get(
      'logger.channel.feeds_enhanced_tokens',
    );
    $this->subscriber = new TokenExpansionSubscriber(
      $this->tokenExpander,
      $logger,
    );
  }

  /**
   * Tests getSubscribedEvents.
   *
   * @covers ::getSubscribedEvents
   */
  public function testGetSubscribedEvents(): void
  {
    $events = TokenExpansionSubscriber::getSubscribedEvents();

    $this->assertArrayHasKey(FeedsEvents::INIT_IMPORT, $events);
    $this->assertArrayHasKey(FeedsEvents::IMPORT_FINISHED, $events);

    $this->assertEquals('onInitImport', $events[FeedsEvents::INIT_IMPORT][0]);
    $this->assertEquals(1000, $events[FeedsEvents::INIT_IMPORT][1]);

    $this->assertEquals(
      'onImportFinished',
      $events[FeedsEvents::IMPORT_FINISHED][0],
    );
    $this->assertEquals(-1000, $events[FeedsEvents::IMPORT_FINISHED][1]);
  }

  /**
   * Tests subscriber service is accessible.
   */
  public function testSubscriberServiceExists(): void
  {
    $this->assertInstanceOf(
      TokenExpansionSubscriber::class,
      $this->subscriber,
    );
  }

  /**
   * Tests feed-level config is NOT expanded to prevent persistence.
   *
   * @covers ::onInitImport
   * @covers ::expandPluginConfigurations
   */
  public function testFeedLevelConfigNotExpanded(): void
  {
    $feed = $this->createMock(FeedInterface::class);
    $feedType = $this->createMock(FeedTypeInterface::class);

    $fetcher = $this->createMock(
      '\Drupal\feeds\Plugin\Type\Fetcher\FetcherInterface',
    );
    $parser = $this->createMock(
      '\Drupal\feeds\Plugin\Type\Parser\ParserInterface',
    );
    $processor = $this->createMock(
      '\Drupal\feeds\Plugin\Type\Processor\ProcessorInterface',
    );

    $feed->method('getType')->willReturn($feedType);
    $feed->method('getFields')->willReturn([]);

    $feedType->method('getFetcher')->willReturn($fetcher);
    $feedType->method('getParser')->willReturn($parser);
    $feedType->method('getProcessor')->willReturn($processor);

    $config = ['host' => 'example.com:[site:name]'];
    $fetcher->method('getConfiguration')->willReturn($config);
    $parser->method('getConfiguration')->willReturn([]);
    $processor->method('getConfiguration')->willReturn([]);

    // Mock expandConfiguration to return same config (simulate no change).
    $mockExpander = $this->createMock(TokenExpander::class);
    $mockExpander->method('expandConfiguration')
      ->willReturn($config);

    // Critical: Feed should NEVER have setConfigurationFor() called.
    // This would persist expanded values to the feed entity.
    $feed->expects($this->never())->method('setConfigurationFor');

    $logger = $this->container->get(
      'logger.channel.feeds_enhanced_tokens',
    );
    $subscriber = new TokenExpansionSubscriber($mockExpander, $logger);

    $event = new InitEvent($feed, 'import');
    $subscriber->onInitImport($event);
  }

  /**
   * Tests cache is cleared after import.
   *
   * @covers ::onImportFinished
   */
  public function testCacheClearedAfterImport(): void
  {
    $feed = $this->createMock(FeedInterface::class);
    $event = $this->createMock(
      '\Drupal\feeds\Event\ImportFinishedEvent',
    );
    $event->method('getFeed')->willReturn($feed);

    // Verify clearCache is called on token expander.
    $mockExpander = $this->createMock(TokenExpander::class);
    $mockExpander->expects($this->once())->method('clearCache');

    $logger = $this->container->get(
      'logger.channel.feeds_enhanced_tokens',
    );
    $subscriber = new TokenExpansionSubscriber($mockExpander, $logger);
    $subscriber->onImportFinished($event);
  }

}
