<?php

declare(strict_types=1);

namespace Drupal\Tests\feeds_enhanced_tokens\Unit;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Utility\Token;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\FeedTypeInterface;
use Drupal\feeds_enhanced_tokens\TokenExpander;
use Drupal\Tests\UnitTestCase;
use Psr\Log\LoggerInterface;

/**
 * Tests the TokenExpander service.
 *
 * @group feeds_enhanced_tokens
 * @coversDefaultClass \Drupal\feeds_enhanced_tokens\TokenExpander
 */
class TokenExpanderTest extends UnitTestCase
{

  /**
   * The token expander service.
   *
   * @var \Drupal\feeds_enhanced_tokens\TokenExpander
   */
  protected TokenExpander $tokenExpander;

  /**
   * Mock token service.
   *
   * @var \Drupal\Core\Utility\Token|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $token;

  /**
   * Mock feed entity.
   *
   * @var \Drupal\feeds\FeedInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $feed;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    $this->token = $this->createMock(Token::class);
    $current_user = $this->createMock(AccountProxyInterface::class);
    $entity_type_manager = $this->createMock(
      EntityTypeManagerInterface::class,
    );
    $logger = $this->createMock(LoggerInterface::class);

    $this->feed = $this->createMock(FeedInterface::class);
    $feed_type = $this->createMock(FeedTypeInterface::class);
    $this->feed->method('getType')->willReturn($feed_type);
    $current_user->method('getAccount')->willReturn(NULL);

    $this->tokenExpander = new TokenExpander(
      $this->token,
      $current_user,
      $entity_type_manager,
      $logger,
    );
  }

  /**
   * Tests expandString with tokens present.
   *
   * @covers ::expandString
   * @covers ::containsTokens
   * @covers ::buildTokenContext
   */
  public function testExpandStringWithTokens(): void
  {
    $text = 'Hello [current-user:name]!';
    $expected = 'Hello Admin!';

    $this->token->expects($this->once())
      ->method('replace')
      ->with($text, $this->anything(), ['clear' => FALSE])
      ->willReturn($expected);

    $result = $this->tokenExpander->expandString($text, $this->feed);

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests expandString without tokens.
   *
   * @covers ::expandString
   * @covers ::containsTokens
   */
  public function testExpandStringWithoutTokens(): void
  {
    $text = 'Plain text without tokens';

    $this->token->expects($this->never())->method('replace');

    $result = $this->tokenExpander->expandString($text, $this->feed);

    $this->assertEquals($text, $result);
  }

  /**
   * Tests expandString with token caching.
   *
   * @covers ::expandString
   * @covers ::containsTokens
   */
  public function testExpandStringCaching(): void
  {
    $text = 'Hello [current-user:name]!';
    $expected = 'Hello Admin!';

    $this->token->expects($this->once())
      ->method('replace')
      ->willReturn($expected);

    $result1 = $this->tokenExpander->expandString($text, $this->feed);
    $result2 = $this->tokenExpander->expandString($text, $this->feed);

    $this->assertEquals($expected, $result1);
    $this->assertEquals($expected, $result2);
  }

  /**
   * Tests expandConfiguration with nested arrays.
   *
   * @covers ::expandConfiguration
   * @covers ::traverseAndExpand
   */
  public function testExpandConfigurationNested(): void
  {
    $config = [
      'host' => 'example.com',
      'port' => '[pantheon_si_tunnel:pantheon_soip_ldap]',
      'nested' => [
        'path' => '/files/[date:custom:Y-m-d]',
        'user' => '[current-user:uid]',
      ],
    ];

    $this->token->method('replace')
      ->willReturnCallback(
        fn($text) => match ($text) {
          '[pantheon_si_tunnel:pantheon_soip_ldap]' => '33061',
          '/files/[date:custom:Y-m-d]' => '/files/2025-01-05',
          '[current-user:uid]' => '1',
          default => $text,
        },
      );

    $result = $this->tokenExpander->expandConfiguration(
      $config,
      $this->feed,
    );

    $this->assertEquals('example.com', $result['host']);
    $this->assertEquals('33061', $result['port']);
    $this->assertEquals('/files/2025-01-05', $result['nested']['path']);
    $this->assertEquals('1', $result['nested']['user']);
  }

  /**
   * Tests clearCache.
   *
   * @covers ::clearCache
   */
  public function testClearCache(): void
  {
    $text = 'Hello [current-user:name]!';
    $expected = 'Hello Admin!';

    $this->token->method('replace')->willReturn($expected);

    $this->tokenExpander->expandString($text, $this->feed);
    $this->tokenExpander->clearCache();

    $this->token->expects($this->once())
      ->method('replace')
      ->willReturn($expected);

    $this->tokenExpander->expandString($text, $this->feed);
  }

}
