<?php

namespace Drupal\feeds_enhanced\Feeds\Parser;

use Drupal\feeds\Exception\EmptyFeedException;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Feeds\Item\DynamicItem;
use Drupal\feeds\Feeds\Parser\ParserBase;
use Drupal\feeds\FeedTypeInterface;
use Drupal\feeds\Result\FetcherResultInterface;
use Drupal\feeds\Result\ParserResult;
use Drupal\feeds\Result\ParserResultInterface;
use Drupal\feeds\StateInterface;

/**
 * Class IniParser
 *  Defines a feed parser for INI files.
 *
 * @package Drupal\feeds_enhanced
 *
 * @FeedsParser(
 *   id = "ini",
 *   title = "INI",
 *   description = @Translation("Parse INI files."),
 *   form = {
 *     "configuration" = "Drupal\feeds_enhanced\Feeds\Parser\Form\IniParserForm",
 *     "feed" = "Drupal\feeds_enhanced\Feeds\Parser\Form\IniParserFeedForm",
 *   },
 * )
 */
class IniParser extends ParserBase {

  /**
   * {@inheritDoc}
   */
  public function parse(
    FeedInterface $feed,
    FetcherResultInterface $fetcher_result,
    StateInterface $state,
  ): ParserResultInterface {
    if (!filesize($fetcher_result->getFilePath())) {
      throw new EmptyFeedException();
    }
    $result = new ParserResult();
    if (!$contents = $this->parseIniFile($fetcher_result->getFilePath())) {
      return $result;
    }
    $item = new DynamicItem();
    $source_map = $this->getSourceMap($feed->getType());
    foreach ($contents as $key => $value) {
      if (array_key_exists($key, $source_map)) {
        $item->set($source_map[$key], $value);
      }
    }
    $result->addItem($item);
    return $result;
  }

  /**
   * {@inheritDoc}
   */
  public function getMappingSources(): array {
    // Rely on user-entered field-mappings.
    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array {
    return [
      static::getMultiModeKey() => FALSE,
    ];
  }

  /**
   * Returns the name of the configuration key that stores the multi-mode
   * configuration value.
   *
   * @return string
   */
  public static function getMultiModeKey(): string {
    return 'ini_mode_multi';
  }

  /**
   * Performs parsing of the target INI file and returns the result as an
   * associative array.
   *
   * @param string $file_path
   *   The path to the INI file to be parsed.
   *
   * @return array
   */
  protected function parseIniFile(string $file_path): array {
    return parse_ini_file(
      $file_path,
      $this->getMultiMode(),
      INI_SCANNER_TYPED
    ) ?: [];
  }

  /**
   * Returns an associative array of field names, keyed by Feed mapping names.
   *
   * @param \Drupal\feeds\FeedTypeInterface $feed_type
   *   The Feed Type whose source mappings should be returned.
   *
   * @return array
   */
  protected function getSourceMap(FeedTypeInterface $feed_type): array {
    $sources = $feed_type->getMappingSources();
    return array_column($sources, 'machine_name', 'value');
  }

  /**
   * Returns the current INI parser read mode.
   *
   * @return string
   */
  protected function getMultiMode(): string {
    return $this->getConfiguration(static::getMultiModeKey()) ?: FALSE;
  }

}
