<?php

namespace Drupal\Tests\feeds_enhanced\Kernel;

use Drupal\Core\Utility\Token;
use Drupal\feeds\Entity\Feed;
use Drupal\feeds\Entity\FeedType;
use Drupal\feeds\StateInterface;
use Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Tests token persistence behavior in SftpFetcher.
 *
 * Verifies critical fix: tokens remain unexpanded in stored feed
 * configuration after fetch operations. Token expansion happens
 * at runtime during fetch but never persists back to database.
 *
 * @group feeds_enhanced
 * @coversDefaultClass \Drupal\feeds_enhanced\Feeds\Fetcher\SftpFetcher
 */
class SftpFetcherTokenPersistenceTest extends KernelTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'feeds',
    'feeds_enhanced',
    'feeds_enhanced_tokens',
    'key',
    'token',
    'node',
    'user',
    'system',
    'field',
    'text',
    'filter',
    'file',
    'options',
  ];

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $tokenService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('feeds_feed');
    $this->installEntitySchema('key');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig([
      'feeds',
      'node',
      'filter',
      'system',
    ]);

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Set site name for token testing (simulates port number).
    $this->config('system.site')->set('name', '41234')->save();

    $this->tokenService = $this->container->get('token');
  }

  /**
   * Tests tokens remain unexpanded in feed config after fetch attempt.
   *
   * This is the critical test for the token persistence fix.
   * Before fix: tokens were expanded and persisted via
   * $feed->setConfigurationFor()
   * After fix: tokens expand at runtime but stay unexpanded in storage.
   *
   * @covers ::fetch
   * @covers ::__construct
   * @covers ::create
   */
  public function testTokensRemainUnexpandedAfterFetch(): void
  {
    $original_host_with_token = '127.0.0.1:[site:name]';

    // Create feed type with token in host.
    $feed_type = FeedType::create([
      'id' => 'persistence_test',
      'label' => 'Token Persistence Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => $original_host_with_token,
        'username' => 'testuser',
        'password' => '',
        'timeout' => 1,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    // Create feed with feed-level override containing token.
    $feed = Feed::create([
      'type' => 'persistence_test',
      'title' => 'Test Feed',
      'source' => '/test/path/file.csv',
    ]);
    $feed->setConfigurationFor($feed_type->getFetcher(), [
      'host' => $original_host_with_token,
      'username' => 'feeduser',
      'password' => '',
      'timeout' => 1,
    ]);
    $feed->save();

    // Verify token expands correctly before fetch.
    $expanded = $this->tokenService->replace(
      $original_host_with_token,
      ['feed' => $feed],
      ['clear' => TRUE],
    );
    $this->assertEquals(
      '127.0.0.1:41234',
      $expanded,
      'Token should expand to port number',
    );

    // Get feed config BEFORE fetch attempt.
    $config_before = $feed->getConfigurationFor(
      $feed_type->getFetcher(),
    );
    $this->assertEquals(
      $original_host_with_token,
      $config_before['host'],
      'Config before fetch should contain unexpanded token',
    );

    // Attempt fetch (will fail due to no SFTP server, but that's OK).
    // The important part is checking config AFTER fetch attempt.
    $state = $this->createMock(StateInterface::class);
    try {
      $feed_type->getFetcher()->fetch($feed, $state);
    }
    catch (\Exception $e) {
      // Expected to fail - we don't have a real SFTP server.
      // We're only testing that config wasn't modified.
    }

    // CRITICAL TEST: Reload feed and verify token is still unexpanded.
    $feed = Feed::load($feed->id());
    $config_after = $feed->getConfigurationFor(
      $feed_type->getFetcher(),
    );

    $this->assertEquals(
      $original_host_with_token,
      $config_after['host'],
      'CRITICAL: Token must remain unexpanded in stored config ' .
      'after fetch. If this fails, expanded values are being ' .
      'persisted to the feed entity.',
    );

    // Also verify feed type config remains unchanged.
    $feed_type = FeedType::load('persistence_test');
    $type_config = $feed_type->getFetcher()->getConfiguration();
    $this->assertEquals(
      $original_host_with_token,
      $type_config['host'],
      'Feed type config should also remain unexpanded',
    );
  }

  /**
   * Tests Token service is properly injected into SftpFetcher.
   *
   * @covers ::__construct
   * @covers ::create
   */
  public function testTokenServiceInjection(): void
  {
    $feed_type = FeedType::create([
      'id' => 'injection_test',
      'label' => 'Token Injection Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'example.com:22',
        'username' => 'user',
        'password' => '',
        'timeout' => 30,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $fetcher = $feed_type->getFetcher();
    $this->assertInstanceOf(
      SftpFetcher::class,
      $fetcher,
      'Fetcher should be SftpFetcher instance',
    );

    // Use reflection to verify token service is injected.
    $reflection = new \ReflectionClass($fetcher);
    $property = $reflection->getProperty('token');
    $property->setAccessible(TRUE);
    $injected_token = $property->getValue($fetcher);

    $this->assertInstanceOf(
      Token::class,
      $injected_token,
      'Token service should be injected into SftpFetcher',
    );
  }

  /**
   * Tests token expansion works in fetch() even without event subscriber.
   *
   * Verifies that SftpFetcher can expand tokens independently,
   * not relying solely on the event subscriber.
   *
   * @covers ::fetch
   */
  public function testTokenExpansionInFetchMethod(): void
  {
    $host_with_token = 'example.com:[site:name]';

    $feed_type = FeedType::create([
      'id' => 'expansion_test',
      'label' => 'Expansion Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => $host_with_token,
        'username' => 'testuser',
        'password' => '',
        'timeout' => 1,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'expansion_test',
      'title' => 'Expansion Test Feed',
      'source' => '/test/file.csv',
    ]);
    $feed->save();

    // Verify token would expand correctly.
    $expanded = $this->tokenService->replace(
      $host_with_token,
      ['feed' => $feed],
      ['clear' => TRUE],
    );
    $this->assertEquals(
      'example.com:41234',
      $expanded,
      'Token should expand correctly for this test',
    );

    // Attempt fetch (will fail due to no SFTP server).
    // The important thing is that fetch() attempts to use the token
    // service and doesn't throw errors related to token expansion.
    $fetcher = $feed_type->getFetcher();
    $state = $this->createMock(StateInterface::class);

    try {
      $fetcher->fetch($feed, $state);
      $this->fail('Expected exception due to no SFTP server');
    }
    catch (\Exception $e) {
      // Expected to fail - no real SFTP server.
      // As long as exception is NOT related to token expansion, test passes.
      $this->assertStringNotContainsString(
        'token',
        strtolower($e->getMessage()),
        'Exception should not be token-related',
      );
    }
  }

  /**
   * Tests backward compatibility with static port numbers.
   *
   * @covers ::fetch
   */
  public function testBackwardCompatibilityWithStaticPorts(): void
  {
    $feed_type = FeedType::create([
      'id' => 'static_port_test',
      'label' => 'Static Port Test',
      'fetcher' => 'sftp',
      'fetcher_configuration' => [
        'host' => 'sftp.example.com:2222',
        'username' => 'testuser',
        'password' => '',
        'timeout' => 1,
      ],
      'parser' => 'csv',
      'processor' => 'entity:node',
      'processor_configuration' => [
        'values' => ['type' => 'article'],
      ],
    ]);
    $feed_type->save();

    $feed = Feed::create([
      'type' => 'static_port_test',
      'title' => 'Static Port Feed',
      'source' => '/test/file.csv',
    ]);
    $feed->save();

    // Get config before fetch.
    $fetcher = $feed_type->getFetcher();
    $config_before = $fetcher->getConfiguration();

    $this->assertEquals(
      'sftp.example.com:2222',
      $config_before['host'],
      'Static port should be stored as-is',
    );

    // Attempt fetch.
    $state = $this->createMock(StateInterface::class);
    try {
      $fetcher->fetch($feed, $state);
    }
    catch (\Exception $e) {
      // Expected to fail.
    }

    // Verify config unchanged.
    $feed = Feed::load($feed->id());
    $feed_type = FeedType::load('static_port_test');
    $config_after = $feed_type->getFetcher()->getConfiguration();

    $this->assertEquals(
      'sftp.example.com:2222',
      $config_after['host'],
      'Static port should remain unchanged after fetch',
    );
  }

}
