<?php

namespace Drupal\feeds_http_oauth\Feeds\Fetcher\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\feeds\FeedInterface;
use Drupal\feeds\Feeds\Fetcher\Form\HttpFetcherFeedForm;

/**
 * Feed-level configuration form for the HTTP OAuth fetcher.
 */
class HttpOAuthFetcherFeedForm extends HttpFetcherFeedForm {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, ?FeedInterface $feed = NULL) {
    // Pull in the core HTTP fetcher feed fields (Feed URL, etc).
    $form = parent::buildConfigurationForm($form, $form_state, $feed);

    $config = $feed
      ? $feed->getConfigurationFor($this->plugin)
      : $this->plugin->defaultFeedConfiguration();

    $has_client_secret = !empty($config['client_secret']);
    $has_password = !empty($config['password']);

    // Put all OAuth fields under plugin[fetcher][...] so core submit logic
    // sees a consistent structure and avoids null-offset warnings.
    $form['plugin']['fetcher']['oauth'] = [
      '#type' => 'details',
      '#title' => $this->t('OAuth 2.0 settings'),
      '#open' => TRUE,
    ];

    $form['plugin']['fetcher']['oauth']['grant_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Grant type'),
      '#description' => $this->t('Select the OAuth 2.0 grant type required by the API provider.'),
      '#options' => [
        'client_credentials' => $this->t('Client Credentials'),
        'password' => $this->t('Password Grant'),
      ],
      '#default_value' => $config['grant_type'] ?? 'client_credentials',
      '#required' => TRUE,
    ];

    $form['plugin']['fetcher']['oauth']['access_token_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Access Token URL'),
      '#default_value' => $config['access_token_url'] ?? '',
      '#maxlength' => 2048,
      '#required' => TRUE,
    ];

    $form['plugin']['fetcher']['oauth']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config['client_id'] ?? '',
      '#required' => TRUE,
    ];

    $form['plugin']['fetcher']['oauth']['client_secret'] = [
      '#type' => 'password',
      '#title' => $this->t('Client Secret'),
      '#description' => $has_client_secret
        ? $this->t('A client secret is already saved. Enter a new value to replace it.')
        : $this->t('Enter the OAuth client secret.'),
      '#required' => !$has_client_secret,
      '#attributes' => $has_client_secret
        ? ['placeholder' => $this->t('(saved)')]
        : [],
    ];

    $form['plugin']['fetcher']['oauth']['scope'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Scope'),
      '#default_value' => $config['scope'] ?? '',
      '#required' => FALSE,
    ];

    // States must target the nested grant_type element name.
    $grant_type_selector = ':input[name="plugin[fetcher][plugin][fetcher][oauth][grant_type]"]';

    $form['plugin']['fetcher']['oauth']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $config['username'] ?? '',
      '#states' => [
        'visible' => [
          $grant_type_selector => ['value' => 'password'],
        ],
        'required' => [
          $grant_type_selector => ['value' => 'password'],
        ],
      ],
    ];

    $form['plugin']['fetcher']['oauth']['password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#description' => $has_password
        ? $this->t('A password is already saved. Enter a new value to replace it.')
        : $this->t('Enter the account password.'),
      '#attributes' => $has_password
        ? ['placeholder' => $this->t('(saved)')]
        : [],
      // Do NOT make this required if a password is already saved. If required,
      // user cannot save other field changes without re-entering the password.
      '#states' => [
        'visible' => [
          $grant_type_selector => ['value' => 'password'],
        ],
        'required' => [
          $grant_type_selector => ['value' => 'password'],
        ],
      ],
    ];

    // If a password already exists, do not require re-entry on every save.
    if ($has_password) {
      unset($form['plugin']['fetcher']['oauth']['password']['#states']['required']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state, ?FeedInterface $feed = NULL) {
    // Capture existing OAuth config BEFORE parent wipes it.
    $existing = $feed->getConfigurationFor($this->plugin);

    // Let parent save the Feed URL and core HTTP fields.
    parent::submitConfigurationForm($form, $form_state, $feed);

    $values = $form_state->getValues();
    $oauth = $values['plugin']['fetcher']['oauth'] ?? [];

    // Start from existing config so secrets are preserved.
    $config = $existing;

    // Overwrite non-secret fields explicitly.
    $config['access_token_url'] = $oauth['access_token_url'] ?? $config['access_token_url'] ?? NULL;
    $config['grant_type'] = $oauth['grant_type'] ?? $config['grant_type'] ?? 'client_credentials';
    $config['client_id'] = $oauth['client_id'] ?? $config['client_id'] ?? NULL;
    $config['scope'] = $oauth['scope'] ?? $config['scope'] ?? NULL;

    // Client secret: overwrite ONLY if non-empty.
    if (!empty($oauth['client_secret'])) {
      $config['client_secret'] = $oauth['client_secret'];
    }

    if (($config['grant_type'] ?? '') === 'password') {
      $config['username'] = $oauth['username'] ?? $config['username'] ?? NULL;

      if (!empty($oauth['password'])) {
        $config['password'] = $oauth['password'];
      }
    }
    else {
      unset($config['username'], $config['password']);
    }

    // Restore merged OAuth config AFTER parent submit.
    $feed->setConfigurationFor($this->plugin, $config);
  }

}
