<?php

namespace Drupal\feeds_http_oauth\Feeds\Fetcher;

use Drupal\feeds\Feeds\Fetcher\HttpFetcher;
use Drupal\feeds\Utility\Feed;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\RequestOptions;

/**
 * Defines an OAuth 2.0-enabled HTTP fetcher.
 *
 * @FeedsFetcher(
 *   id = "http_oauth",
 *   title = @Translation("Download from url (OAuth 2.0)"),
 *   description = @Translation("Downloads data from a URL using OAuth 2.0 to obtain an access token."),
 *   form = {
 *     "configuration" = "Drupal\feeds\Feeds\Fetcher\Form\HttpFetcherForm",
 *     "feed" = "Drupal\feeds_http_oauth\Feeds\Fetcher\Form\HttpOAuthFetcherFeedForm"
 *   }
 * )
 */
class HttpOAuthFetcher extends HttpFetcher {

  /**
   * {@inheritdoc}
   */
  protected function get($url, $sink, $cache_key = FALSE, array $options = [], $feed = NULL) {
    if (!$feed) {
      throw new \RuntimeException('OAuth fetcher requires a feed context.');
    }

    $feed_config = $feed->getConfigurationFor($this);

    // OAuth is mandatory for this fetcher.
    $token = $this->getAccessToken($feed_config);

    if (
      empty($token)
      || empty($token->access_token)
      || empty($token->token_type)
    ) {
      throw new \RuntimeException('Failed to retrieve OAuth access token.');
    }

    $url = Feed::translateSchemes($url);

    $options = [
      RequestOptions::SINK => $sink,
      RequestOptions::HEADERS => [
        'Authorization' => $token->token_type . ' ' . $token->access_token,
      ],
    ];

    // Add User-Agent header from the default guzzle client config if present.
    if (isset($this->client->getConfig('headers')['User-Agent'])) {
      $options[RequestOptions::HEADERS]['User-Agent'] =
        $this->client->getConfig('headers')['User-Agent'];
    }

    // Add cached headers if requested.
    if ($cache_key && ($cache = $this->cache->get($cache_key))) {
      if (isset($cache->data['etag'])) {
        $options[RequestOptions::HEADERS]['If-None-Match'] =
          $cache->data['etag'];
      }
      if (isset($cache->data['last-modified'])) {
        $options[RequestOptions::HEADERS]['If-Modified-Since'] =
          $cache->data['last-modified'];
      }
    }

    try {
      $response = $this->client->getAsync($url, $options)->wait();
    }
    catch (RequestException $e) {
      throw new \RuntimeException(
        'The feed from %site seems to be broken because of error ' . $e->getMessage()
      );
    }

    if ($cache_key) {
      $this->cache->set(
        $cache_key,
        array_change_key_case($response->getHeaders())
      );
    }

    return $response;
  }

  /**
   * Retrieves an OAuth access token.
   *
   * @param array $feed_config
   *   The feed configuration expected.
   *
   * @return \stdClass
   *   The oauth access token.
   */
  protected function getAccessToken($feed_config):\stdClass {
    if (empty($feed_config['access_token_url'])) {
      throw new \RuntimeException('Missing OAuth access token URL.');
    }

    $options = [
      RequestOptions::HEADERS => [
        'Content-Type' => 'application/x-www-form-urlencoded',
      ],
      RequestOptions::FORM_PARAMS => [],
    ];

    if ($feed_config['grant_type'] === 'client_credentials') {
      $options[RequestOptions::FORM_PARAMS] = [
        'grant_type' => 'client_credentials',
        'client_id' => $feed_config['client_id'] ?? '',
        'client_secret' => $feed_config['client_secret'] ?? '',
      ];

      if (!empty($feed_config['scope'])) {
        $options[RequestOptions::FORM_PARAMS]['scope'] = $feed_config['scope'];
      }
    }
    elseif ($feed_config['grant_type'] === 'password') {
      $basic = base64_encode(
        ($feed_config['client_id'] ?? '') . ':' . ($feed_config['client_secret'] ?? '')
      );

      $options[RequestOptions::HEADERS]['Authorization'] = 'Basic ' . $basic;
      $options[RequestOptions::FORM_PARAMS] = [
        'grant_type' => 'password',
        'username' => $feed_config['username'] ?? '',
        'password' => $feed_config['password'] ?? '',
      ];

      if (!empty($feed_config['scope'])) {
        $options[RequestOptions::FORM_PARAMS]['scope'] = $feed_config['scope'];
      }
    }
    else {
      throw new \RuntimeException('Unsupported OAuth grant type.');
    }

    try {
      $response = $this->client->request(
        'POST',
        $feed_config['access_token_url'],
        $options
      );
    }
    catch (RequestException $e) {
      watchdog_exception('feeds_http_oauth', $e);
      throw new \RuntimeException('OAuth token request failed.');
    }

    $token = json_decode($response->getBody()->getContents(), FALSE);

    if (!is_object($token)) {
      throw new \RuntimeException('Invalid OAuth token response.');
    }

    return $token;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    $config = parent::defaultConfiguration();
    $config['auto_detect_feeds'] = FALSE;
    $config['use_pubsubhubbub'] = FALSE;
    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultFeedConfiguration() {
    return [
      'access_token_url' => NULL,
      'grant_type' => 'client_credentials',
      'client_id' => NULL,
      'client_secret' => NULL,
      'scope' => NULL,
      'username' => NULL,
      'password' => NULL,
    ];
  }

}
