<?php

namespace Drupal\feide_login\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\externalauth\AuthmapInterface;
use Drupal\externalauth\ExternalAuthInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\user\UserInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Psr\Log\LoggerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\key\KeyRepositoryInterface;

/**
 * Helper class to make login easy.
 */
class FeideHelper {

  use StringTranslationTrait;

  /**
   * API Endpoint.
   *
   * @var string
   */
  protected $apiBase = 'https://auth.dataporten.no';

  /**
   * A configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * FeideHelper constructor.
   *
   * @param ConfigFactoryInterface     $config_factory
   *   The configuration factory.
   * @param ClientInterface            $httpClient
   *   The HTTP client.
   * @param ExternalAuthInterface      $externalauth
   *   The ExternalAuth Service.
   * @param AuthmapInterface           $authmap
   *   The AuthMap Service.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param KeyRepositoryInterface     $keyRepository
   *   The key repository service.
   * @param LoggerInterface            $logger
   *   A logger instance.
   * @param MessengerInterface         $messenger
   *   The messenger.
   * @param ModuleHandlerInterface     $moduleHandler
   * @param TranslationInterface       $stringTranslation
   * @param FileRepositoryInterface    $fileRepository
   * @param FileSystemInterface        $fileSystem
   */
  public function __construct(ConfigFactoryInterface $config_factory,
                              protected ClientInterface $httpClient,
                              protected ExternalAuthInterface $externalauth,
                              protected AuthmapInterface $authmap,
                              protected EntityTypeManagerInterface $entityTypeManager,
                              protected KeyRepositoryInterface $keyRepository,
                              protected LoggerInterface $logger,
                              protected MessengerInterface $messenger,
                              protected ModuleHandlerInterface $moduleHandler,
                              TranslationInterface $string_translation,
                              protected FileRepositoryInterface $fileRepository,
                              protected FileSystemInterface $fileSystem) {
    $this->config = $config_factory->get('feide_login.settings');
    $this->stringTranslation = $string_translation;
  }

  /**
   * Debug enabled.
   *
   * Checks weather debug is enabled.
   *
   * @return bool
   */
  public function debugEnabled() {
    return $this->config->get('debug');
  }

  /**
   * Retrieves an existing user by their name or email.
   * Note that we're no longer using attributes here, as
   *    they changed and it was hard to get a fixed value here.
   *
   * @param string $authname The name of the user.
   * @param array $attributes The attributes of the user, including the email.
   * @return \Drupal\Core\Entity\EntityInterface|bool The existing user entity,
   *   or FALSE if not found.
   */
  public function getExistingUser($authname, $attributes) {
    // Note refactor: loading by attributes proved hard, as the attributes change after login. Now
    // Loading users by authname only
    $users_by_mail = user_load_by_mail($authname);

    if (!empty($users_by_mail)) {
      return $users_by_mail;
    }
    if ($this->debugEnabled()) {
      $this->logger->notice('Found no existing users with mail: %authname, trying with username next', [
        '%email' => $authname,
      ]);
    }

    // Attempt to load user by username.
    $user_by_name = user_load_by_name($authname);
    if ($user_by_name) {
      return $user_by_name;
    }

    // Logging if we find no users by name or mail
    if ($this->debugEnabled()) {
      $this->logger->notice('Found no existing users with authname: %name', [
        '%name' => $authname,
      ]);
    }

    // Return FALSE if no user is found.
    return FALSE;
  }

  /**
   * Generate the url for authentication.
   *
   * @return string
   */
  public function getAuthorizationRequest(): string {
    $key_repo = $this->config->get('key_repo');
    $credentials = $this->keyRepository->getKey($key_repo)->getKeyValues();
    $client_id = $credentials['client_id'];
    $redirect_uri = $credentials['redirect_uri'];

    $url = Url::fromUri($this->apiBase . '/oauth/authorization', [
      'query' => [
        'client_id' => $client_id,
        'response_type' => 'code',
        'redirect_uri' => $redirect_uri,
      ],
    ]);

    return $url->toUriString();
  }

  /**
   * Fetch the token.
   */
  public function getOauthToken($code) {
    $key_repo = $this->config->get('key_repo');
    $credentials = $this->keyRepository->getKey($key_repo)->getKeyValues();
    $client_id = $credentials['client_id'];
    $client_secret = $credentials['client_secret'];
    $redirect_uri = $credentials['redirect_uri'];

    $params = [
      'grant_type' => 'authorization_code',
      'code' => $code,
      'client_id' => $client_id,
      'redirect_uri' => $redirect_uri,
    ];

    $response = $this->httpClient->post($this->apiBase . '/oauth/token', [
      'body' => http_build_query($params),
      'headers' => [
        'Authorization' => 'Basic ' . base64_encode($client_id . ':' . $client_secret),
        'Content-Type' => 'application/x-www-form-urlencoded',
      ],
    ]);

    $body = $response->getBody();
    $data = $body->getContents();
    return json_decode($data, TRUE);
  }

  /**
   * Get user info.
   */
  public function getUserInfo($token) {
    $response = $this->httpClient->get($this->apiBase . '/openid/userinfo', [
      'headers' => [
        'Authorization' => 'Bearer ' . $token,
      ],
    ]);

    $body = $response->getBody();
    $data = $body->getContents();

    return json_decode($data, TRUE);
  }

  /**
   * Log in and optionally register a user based on the authname provided.
   *
   * @param string $authname
   *   The authentication name.
   *
   * @param array $data
   *   Info about token and user details.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The logged in Drupal user.
   */
  public function handleAuthentication($authname, $data) {
    // Step 1: Attempt to log in with existing credentials, if user is already linked

    if ($this->debugEnabled()) {
      $this->logger->debug('Attempting to handle auth with authname "%authname" and data <pre>%data</pre>', [
        '%authname' => $authname,
        '%data' => print_r($data, TRUE),
      ]);
    }

    $account = $this->externalauth->login($authname, 'feide_login');

    // If login is successful, return the account immediately.
    if ($account) {
      return $account;
    }

    // Step 2: If login fails, attempt to register a new user if allowed.
    if ($this->config->get('register_users')) {
      $account = $this->registerNewUser($authname, $data);
      if ($account) {
        // Register successful, finalize the login.
        $account = $this->externalauth->userLoginFinalize($account, $authname, 'feide_login');
        // Save the new token after registration and login finalization.
        $this->authmap->save($account, 'feide_login', $authname, $data['token_data']);
        return $account;
      }
    }

    // Step 3: If registration fails, attempt to link an existing account if allowed.
    if ($this->config->get('link_accounts')) {
      $account = $this->linkExistingAccount($authname, $data);
      if ($account) {
        // Linking successful, finalize the login.
        $account = $this->externalauth->userLoginFinalize($account, $authname, 'feide_login');
        // Save the new token after linking and login finalization.
        $this->authmap->save($account, 'feide_login', $authname, $data['token_data']);
        return $account;
      }
    }

    // Step 4: If no account was found or successfully authenticated, log an error if debugging is enabled.
    if ($this->debugEnabled()) {
      $this->logger->error('Failed to login, register, or link user with authname "%authname".', [
        '%authname' => $authname,
      ]);
    }

    return null;
  }

  /**
   * Registers a user locally as one authenticated by the SimpleSAML IdP.
   *
   * @param string $authname
   *   The authentication name.
   *
   * @return \Drupal\Core\Entity\EntityInterface|bool
   *   The registered Drupal user.
   *
   * @throws \Exception
   *   An ExternalAuth exception.
   */
  public function externalRegister($authname, $data) {
    // Check if linking accounts is enabled.
    if ($this->config->get('link_accounts')) {
      $account = $this->linkExistingAccount($authname, $data);
      if ($account) {
        if ($this->debugEnabled()) {
          $this->logger->debug('Linked authname "%authname" to existing Drupal user with ID %id.', [
            '%authname' => $authname,
            '%id' => $account->id(),
          ]);
        }
        return $this->externalauth->userLoginFinalize($account, $authname, 'feide_login');
      }
    }

    // Check if registration of new users is allowed.
    if ($this->config->get('register_users')) {
      $account = $this->registerNewUser($authname, $data);
      if ($account) {
        return $this->externalauth->userLoginFinalize($account, $authname, 'feide_login');
      }
    }

    // We're not allowed to link or register new users on the site.
    $this->messenger->addMessage($this->t('We are sorry. While you have successfully authenticated, you are not yet entitled to access this site. Please ask the site administrator to provision access for you.'), 'status');
    return FALSE;
  }

  /**
   * Links an existing user to a SimpleSAML account.
   *
   * @param string $authname
   *   The authentication name.
   * @param array $data
   *   The user data from the IdP.
   *
   * @return \Drupal\Core\Entity\EntityInterface|bool
   *   The linked Drupal user or FALSE if linking failed.
   */
  public function linkExistingAccount($authname, $data) {
    // Load user by name or e-mail.
    $user = $this->getExistingUser($authname, $data);

    if ($user && $this->config->get('link_accounts')) {

      if ($this->debugEnabled()) {
        $this->logger->debug('Linking authname %authname to existing Drupal user with ID %id.', [
          '%authname' => $authname,
          '%id' => $user->id(),
        ]);
      }

      $this->externalauth->linkExistingAccount($authname, 'feide_login', $user);
      return $user;
    }

    return FALSE;
  }


  /**
   * Registers a new user authenticated by the SimpleSAML IdP.
   *
   * @param string $authname
   *   The authentication name.
   * @param array $data
   *   The user data from the IdP.
   *
   * @return \Drupal\Core\Entity\EntityInterface|bool
   *   The registered Drupal user or FALSE if registration failed.
   */
  public function registerNewUser($authname, $data) {
    try {
      // Prepare the account data array.
      $account_data = [
        'name' => $authname,
        'mail' => $data['user_info']['email'],
      ];

      // Register the new user with the account data.
      $account = $this->externalauth->register($authname, 'feide_login', $account_data);
      return $account;
    }
    catch (\Exception $ex) {
      $this->logger->error('Error registering user: @message', ['@message' => $ex->getMessage()]);
      $this->messenger->addMessage($this->t('Error registering user: An account with this username already exists.'), 'error');
      return FALSE;
    }
  }

  /**
   * Check if login is allowed.
   */
  public function allowLogin($authname, $attributes = []) {
    $results = $this->moduleHandler->invokeAll('feide_login_allow_login', [$authname, $attributes]);

    foreach ($results as $result) {
      if ($result === FALSE) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * List a users groups if they have permission to list groups.
   *
   * @param $uid
   *
   * @return array
   */
  public function listUserGroups($uid) {
    $authData = $this->authmap->getAuthData($uid, 'feide_login');
    $token_info = unserialize($authData['data']);

    if (strpos($token_info['scope'], 'groups') === FALSE) {
      // User has not access to list groups, no need to continue.
      return [];
    }

    $cache = \Drupal::cache();
    $cached_data = $cache->get('feide:groups:' . $uid);
    if ($cached_data) {
      return $cached_data->data;
    }

    $response = $this->httpClient->get('https://groups-api.dataporten.no/groups/me/groups', [
      'headers' => [
        'Authorization' => 'Bearer ' . $token_info['access_token'],
      ],
    ]);

    $body = $response->getBody();
    $data = json_decode($body->getContents(), TRUE);
    $groups = [];

    foreach ($data as $group) {
      $members = $this->listGroupMembers($uid, $group['id']);

      if (count($members) === 0) {
        // Skip empty lists.
        continue;
      }

      $groups[$group['id']] = $group;
      $groups[$group['id']]['members'] = $members;
    }

    // Cache results for one hour:
    $cache->set('feide:groups:' . $uid, $groups, time() + 3600);

    return $groups;
  }

  public function listGroupMembers($uid, $groupId) {
    $authData = $this->authmap->getAuthData($uid, 'feide_login');
    $token_info = unserialize($authData['data']);

    if (strpos($token_info['scope'], 'groups') === FALSE) {
      // User has not access to list groups, no need to continue.
      return FALSE;
    }

    $response = $this->httpClient->get('https://groups-api.dataporten.no/groups/groups/' . rawurlencode($groupId) . '/members', [
      'headers' => [
        'Authorization' => 'Bearer ' . $token_info['access_token'],
      ],
    ]);

    $body = $response->getBody();
    $data = json_decode($body->getContents(), TRUE);

    return $data;
  }

}
