<?php

namespace Drupal\feide_login\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\feide_login\Service\FeideHelper;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Session\AccountInterface;
use Psr\Log\LoggerInterface;

/**
 * LoginController.
 */
class LoginController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * Setup dependencies.
   *
   * @param \Drupal\feide_login\Service\FeideHelper $feideHelper
   *   The Feide Login helper service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current account.
   * @param \Psr\Log\LoggerInterface $logger
   *    The logger service.
   */
  public function __construct(public FeideHelper $feideHelper,
                              public RequestStack $requestStack,
                              protected AccountInterface $account,
                              protected LoggerInterface $logger
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('feide_login.helper'),
      $container->get('request_stack'),
      $container->get('current_user'),
      $container->get('logger.factory')->get('feide_login'),
    );
  }

  /**
   * Login with IDPorten.
   */
  public function authenticate() {

    if ($this->account->isAuthenticated()) {
      return $this->redirect('user.login');
    }

    $redirect = new TrustedRedirectResponse($this->feideHelper->getAuthorizationRequest());

    return $redirect;
  }

  /**
   * Handle the redirect from Feide.
   */
  public function feideResponse() {
    $request = $this->requestStack->getCurrentRequest();
    $code = $request->query->get('code');

    $token_data = $this->feideHelper->getOauthToken($code);
    $user_data = $this->feideHelper->getUserInfo($token_data['access_token']);
//  $authname = str_replace('feide:', '', $user_data['dataporten-userid_sec'][0]);
//  $authname = $user_data['https://n.feide.no/claims/eduPersonPrincipalName'];
    $authname = $user_data['email'];

    $data = [
      'token_data' => $token_data,
      'user_info' => $user_data,
    ];

    if ($this->feideHelper->debugEnabled()) {
      $this->logger->debug('<pre>Feide response: @data </pre>', [
        '@data' => print_r(array_merge($data, ['authname' => $authname]), TRUE),
      ]);
    }

    // Check if login is allowed.
    if (!$this->feideHelper->allowLogin($authname, $data['user_info'])) {
      if ($this->feideHelper->debugEnabled()) {
        $this->logger->debug('Login attempt denied for user: @authname', ['@authname' => $authname]);
      }

      $this->messenger->addError('You are not allowed to login with this service.');
      return $this->redirect('user.login');
    }

    $this->feideHelper->handleAuthentication($authname, $data);

    return $this->redirect('<front>');
  }

}
