<?php

declare(strict_types=1);

namespace Drupal\field_access;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Handler that determines access to a field based on permission maps.
 *
 * Permission maps are arrays that detail which users should be granted access
 * to which fields for a given entity type and bundle. The array items of the
 * map must be keyed by the machine name of each field; if a field is not
 * defined in the map, a neutral result is returned i.e. not within the
 * responsibility of this handler to control access for.
 *
 * Each field item may have a value in one of the following formats.
 * - A boolean `FALSE`, which indicates that access to the field should be
 *   denied to all users for all operations.
 * - An array that details which users should be granted access to which
 *   operations on the field, based on their roles.
 *
 * In the latter case, the array items must be keyed by the ID of the operation
 * (view/create/edit); if an operation is not defined, a neutral result is
 * returned i.e. not within the responsibility of this handler to control access
 * for.
 *
 * Each field or operation item may have a value in the following formats.
 * - A boolean `TRUE`; indicates that access to the field should be granted to
 *   users with any role.
 * - A boolean `FALSE`; indicates that access to the field should be denied to
 *   all users.
 * - An array containing a list of user roles; only users that have one or more
 *   of these roles will be granted access to the operation.
 *
 * Note that setting a value to `TRUE` and not setting the key at all is the
 * same i.e. users are granted access. The ability to set to `TRUE` is supported
 * in case a more explicit syntax is preferred for better visibility.
 *
 * Also, note that granting access to a user is done by returning a neutral
 * result. The user may still be denied access to the field/operation by another
 * implementation.
 *
 * Also, note that access to users with the `administrator` role is granted to
 * all fields and operations. The permission map is not taken into account at
 * all in that case and it is not necessary to define permissions for that role
 * in the maps - they won't have any effect.
 *
 * Example of a permission map for the entity bundle `article` that has a
 * `title` and a `body` field:
 *
 * @code
 * const ARTICLE = [
 *   'title' => FALSE,
 *   'body' => [
 *     'create' => ['content_editor'],
 *     'edit' => FALSE,
 *   ],
 * ]
 * @endcode
 *
 * The above will result in the following:
 * - Users with the `administrator` role have access to all operations on both
 *   fields.
 * - The `title` field cannot be viewed, created or edited by anybody other than
 *   users with the `administrator` role.
 * - The `body` field can be viewed by all users.
 * - The `body` field can be created by users with the `content_editor` role.
 * - The `body` field cannot be edited by anybody other than users with the
 *   `administrator` role.
 *
 * Note that Drupal core does not provide a `create` operation, only `edit`. In
 * this context, `create` is the ability to set the value of the field when a
 * new entity is created.
 *
 * @see hook_entity_field_access()
 *
 * @I Support user IDs
 *    type     : improvement
 *    priority : normal
 *    labels   : map
 *    notes    : With the assumption that a user role will never be a numeric,
 *               integer value, we can mix and match user IDs with user roles in
 *               the same array.
 * @I Support access result mode
 *    type     : improvement
 *    priority : normal
 *    labels   : map
 *    notes    : We currently return a neutral result; add a constant that can
 *               instruct the handler to return an allowed result.
 */
interface AccessHandlerInterface {

  /**
   * Returns the access result for the given field, account and operation.
   *
   * @param array $map_classes
   *   An array keyed by the entity type ID and containing the class that holds
   *   the permission maps for its bundles.
   * @param string $operation
   *   The operation to check access for.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to check access for.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check access for.
   * @param \Drupal\Core\Field\FieldItemListInterface|null $items
   *   The entity field object for which to check access, or `NULL` if access is
   *   checked for the field definition without any specific value available.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   *
   * @throws \InvalidArgumentException
   *   When the resolved permission map is not in the expected format.
   */
  public function access(
    array $map_classes,
    string $operation,
    FieldDefinitionInterface $field_definition,
    AccountInterface $account,
    ?FieldItemListInterface $items = NULL,
  ): AccessResultInterface;

}
