<?php

declare(strict_types=1);

namespace Drupal\field_access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Default implementation of the access handler.
 */
class AccessHandler implements AccessHandlerInterface {

  /**
   * Constructs a new AccessHandler object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function access(
    array $map_classes,
    string $operation,
    FieldDefinitionInterface $field_definition,
    AccountInterface $account,
    ?FieldItemListInterface $items = NULL,
  ): AccessResultInterface {
    $entity = $items === NULL ? NULL : $items->getEntity();

    $map = $this->permissionMap($entity, $field_definition, $map_classes);
    if (!$map) {
      return AccessResult::neutral();
    }

    // Administrators bypass this access control; they will be granted access
    // to all fields unless other implementations deny access explicitly.
    $roles = $account->getRoles();
    if (\in_array('administrator', $roles, TRUE)) {
      return AccessResult::neutral();
    }

    $field_name = $field_definition->getName();

    return $this->fieldLevelResult($map, $roles, $field_name) ??
      $this->operationLevelResult($map, $roles, $entity, $field_name, $operation);
  }

  /**
   * Returns the permission map for the given field.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface|null $entity
   *   The parent entity.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to get the permission map for.
   * @param array $map_classes
   *   An array keyed by the entity type ID and containing the class that holds
   *   the permission maps for its bundles.
   *
   * @return array
   *   The permission map for the entity type/bundle that the field belongs to.
   */
  protected function permissionMap(
    ?FieldableEntityInterface $entity,
    FieldDefinitionInterface $field_definition,
    array $map_classes,
  ): array {
    $entity_type_id = $field_definition->getTargetEntityTypeId();
    if (!\in_array($entity_type_id, \array_keys($map_classes), TRUE)) {
      return [];
    }

    $class = $map_classes[$entity_type_id];
    $bundle = $field_definition->getTargetBundle();
    // Base field definitions do not have a target bundle. Get the bundle from
    // the entity in that case.
    if (!$bundle && $entity !== NULL) {
      $bundle_entity_type = $this->entityTypeManager
        ->getDefinition($entity->getEntityTypeId())
        ->getBundleEntityType();
      $bundle = $bundle_entity_type ? $entity->bundle() : NULL;
    }

    $bundle = $bundle ? strtoupper($bundle) : '';
    if (!$bundle || !defined("{$class}::{$bundle}")) {
      $constant = "{$class}::__DEFAULT";
      return defined($constant) ? constant($constant) : [];
    }

    return constant("{$class}::{$bundle}");
  }

  /**
   * Returns the field-level access result for the given field.
   *
   * @param array $map
   *   The permission map applicable to the field.
   * @param string[] $roles
   *   An array containing the IDs of the roles for the user.
   * @param string $field_name
   *   The machine name of the field being evaluated.
   *
   * @return \Drupal\Core\Access\AccessResultInterface|null
   *   The access result, or `NULL` if we don't have a result configured at the
   *   field level.
   *
   * @throws \InvalidArgumentException
   *   When the permission map is not in the expected format.
   */
  protected function fieldLevelResult(
    array $map,
    array $roles,
    string $field_name,
  ): ?AccessResultInterface {
    // If the permissions for the field are not defined in the map, or they are
    // defined as `NULL`, it inherits the permissions from the parent entity.
    if (!isset($map[$field_name])) {
      return AccessResult::neutral();
    }

    // If we have a boolean or string then we can respond at the field level
    // regardless of the operation.
    if (is_bool($map[$field_name])) {
      return $this->booleanResult($map[$field_name]);
    }
    if (is_string($map[$field_name])) {
      return $this->regexResult($map[$field_name], $roles);
    }

    if (!\is_array($map[$field_name])) {
      throw new \InvalidArgumentException(sprintf(
        'Permission map item for field "%s" must hold boolean, string or array, "%s" given.',
        $field_name,
        gettype($map[$field_name]),
      ));
    }

    // Not a field-level map.
    if (!is_int(key($map[$field_name]))) {
      return NULL;
    }

    // If we have a numeric array, it must contain the user roles that are
    // granted access to the operation. If the user has one of the roles then we
    // return a neutral result so that the field inherits access from the parent
    // entity i.e. if the user has access to view/create/edit the entity then
    // has access to do the same to the field - unless another access control
    // handler forbids. If the user does not have at least one of the roles then
    // we forbid.
    if (!\array_intersect($map[$field_name], $roles)) {
      return AccessResult::forbidden();
    }

    return AccessResult::neutral();
  }

  /**
   * Returns the operation-level access result for the given operation.
   *
   * Unlike at the field level, we always have a result at the operation level
   * even if that is a neutral result.
   *
   * @param array $map
   *   The permission map applicable to the field.
   * @param string[] $roles
   *   An array containing the IDs of the roles for the user.
   * @param \Drupal\Core\Entity\FieldableEntityInterface|null $entity
   *   The parent entity.
   * @param string $field_name
   *   The machine name of the field being evaluated.
   * @param string $operation
   *   The operation.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   *
   * @throws \InvalidArgumentException
   *   When the permission map is not in the expected format.
   */
  protected function operationLevelResult(
    array $map,
    array $roles,
    ?FieldableEntityInterface $entity,
    string $field_name,
    string $operation,
  ): ?AccessResultInterface {
    // We differentiate between entity creation and update. There are cases
    // where fields can be given values when the parent entity is first created
    // but cannot be edited thereafter.
    if ($operation === 'edit' && $entity && $entity->isNew()) {
      $operation = 'create';
    }

    // No value defined for the operation, or defined as `NULL`, inherit the
    // permissions from the parent entity.
    if (!isset($map[$field_name][$operation])) {
      return AccessResult::neutral();
    }

    if (is_bool($map[$field_name][$operation])) {
      return $this->booleanResult($map[$field_name][$operation]);
    }
    if (is_string($map[$field_name][$operation])) {
      return $this->regexResult($map[$field_name][$operation], $roles);
    }

    if (!\is_array($map[$field_name][$operation])) {
      throw new \InvalidArgumentException(sprintf(
        'Permission map item for operation "%s" for field "%s" must be boolean, string or array, "%s" given.',
        $operation,
        $field_name,
        gettype($map[$field_name][$operation]),
      ));
    }

    // Otherwise, if we have a value it must contain the user roles that are
    // granted access to the operation. If the user has one of the roles then we
    // return a neutral result so that the field inherits access from the parent
    // entity i.e. if the user has access to view/create/edit the entity then
    // has access to do the same to the field - unless another access control
    // handler forbids. If the user does not have at least one of the roles then
    // we forbid.
    if (!\array_intersect($map[$field_name][$operation], $roles)) {
      return AccessResult::forbidden();
    }

    return AccessResult::neutral();
  }

  /**
   * Returns the access result for the given boolean value.
   *
   * Whether we are the field level or at the operation level, `TRUE` evaluates
   * to a neutral access result so that the final access is determined either by
   * the parent entity access or by other field access implementations. `FALSE`
   * evaluates to access forbidden.
   *
   * @param bool $value
   *   The boolean to evaluate.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function booleanResult(bool $value): AccessResultInterface {
    if ($value === TRUE) {
      return AccessResult::neutral();
    }

    return AccessResult::forbidden();
  }

  /**
   * Returns the access result for the given regular expression pattern.
   *
   * The regular expression is evaluated against the user roles. If there's at
   * least one role matching the pattern then the result is neutral
   * (i.e. inherit from the parent entity), otherwise forbidden.
   *
   * @param string $pattern
   *   The regex pattern.
   * @param string[] $roles
   *   An array containing the IDs of the roles for the user.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   *
   * @throws \InvalidArgumentException
   *   When evaluating the regular expression fails, likely due to the pattern
   *   provided being invalid or unsupported.
   *
   * @see preg_grep()
   * @link https://www.php.net/manual/en/function.preg-grep.php
   */
  protected function regexResult(
    string $pattern,
    array $roles,
  ): AccessResultInterface {
    $result = preg_grep($pattern, $roles);
    if ($result === FALSE) {
      throw new \InvalidArgumentException(sprintf(
        'Failed to run regex matching of user roles against "%s", likely bad pattern.',
        $pattern,
      ));
    }

    return count($result) === 0 ?
      AccessResult::forbidden() :
      AccessResult::neutral();
  }

}
