<?php

namespace Drupal\Tests\field_access\Unit;

use Drupal\field_access\AccessHandler;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Tests\UnitTestCase;

use Prophecy\Argument;

/**
 * @coversDefaultClass \Drupal\field_access\AccessHandler
 *
 * @group contrib
 * @group field_access
 * @group unit
 */
class AccessHandlerTest extends UnitTestCase {

  use AccessHandlerTestTrait;

  /**
   * Tests that the result is neutral when there is no map for the field.
   *
   * @covers ::access
   */
  public function testNoMap() {
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->shouldNotBeCalled();

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // Different entity type.
    $field_definition = $this->prophesizeFieldDefinition('user', NULL);
    $field_definition
      ->getName()
      ->shouldNotBeCalled();

    $result = $handler->access(
      [
        'node' => NodeTestPermissionMap::class,
      ],
      'view',
      $field_definition->reveal(),
      $account->reveal(),
      NULL,
    );
    $this->assertEquals(TRUE, $result->isNeutral());

    // No target bundle and no entity.
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->shouldNotBeCalled();

    $result = $handler->access(
      [
        'node' => NodeTestPermissionMapNoDefault::class,
      ],
      'view',
      $field_definition->reveal(),
      $account->reveal(),
      NULL,
    );
    $this->assertEquals(TRUE, $result->isNeutral());
  }

  /**
   * Tests that the result is neutral when we have a boolean allow at the field
   * level, defined by the default permission map.
   *
   * @covers ::access
   */
  public function testDefaultAllow() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('created');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['authenticated']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations allowed.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
  }

  /**
   * Tests that the result is neutral when we do not have a value defined by the
   * default permission map.
   *
   * @covers ::access
   */
  public function testDefaultUndefined() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('undefined_field');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['authenticated']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations allowed.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
  }

  /**
   * Tests that the result is forbidden when we have a boolean deny at the field
   * level, defined by the bundle permission map.
   *
   * @covers ::access
   */
  public function testBundleDenied() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['authenticated']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations allowed.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isForbidden());
    }
  }

  /**
   * Tests that the result is neutral when we have mixed syntax at the operation
   * level, defined by the bundle permission map.
   *
   * @covers ::access
   */
  public function testBundleMixedAllowed() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('title');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['content_editor']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations allowed, by role, by boolean, or by ommission.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
  }

  /**
   * Tests that the result is forbidden when we have mixed syntax at the
   * operation level, defined by the bundle permission map.
   *
   * @covers ::access
   */
  public function testBundleMixedDenied() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('body');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['authenticated']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // - `view` operation forbidden by boolean syntax.
    // - `create` operation forbidden by role syntax.
    foreach (['view', 'create'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isForbidden());
    }
  }

  /**
   * Tests that the result is neutral when we have role syntax at the field
   * level, defined by the bundle permission map.
   *
   * @covers ::access
   */
  public function testBundleFieldLevelRolesAllowed() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('uid');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['content_editor']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations allowed, by role.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
  }

  /**
   * Tests that the result is forbidden when we have role syntax at the field
   * level, defined by the bundle permission map.
   *
   * @covers ::access
   */
  public function testBundleFieldLevelRolesDenied() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('uid');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['authenticated']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // All operations forbidden, by role.
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isForbidden());
    }
  }

  /**
   * Tests that the result is forbidden based on a regular expression when a
   * field-level map item is provided as a string.
   *
   * @covers ::access
   */
  public function testBundleFieldLevelRegexDenied() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we mock the result of
    // `preg_grep`.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_grep` returns an empty array (no match), access forbidden.
    $preg_grep = $this->mockPregGrep([]);
    $preg_grep->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isForbidden());
    }
    $preg_grep->disable();
  }

  /**
   * Tests that the result is allowed based on a regular expression when a
   * field-level map item is provided as a string.
   *
   * @covers ::access
   */
  public function testBundleFieldLevelRegexAllowed() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we mock the result of
    // `preg_grep`.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_grep` returns a non-empty array (one or more roles matched),
    // allowed.
    $preg_grep = $this->mockPregGrep(['anonymous']);
    $preg_grep->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
    $preg_grep->disable();
  }

  /**
   * Tests that the result is forbidden based on a regular expression when a
   * operation-level map item is provided as a string.
   *
   * @covers ::access
   */
  public function testBundleOperationLevelRegexDenied() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('created');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we mock the result of
    // `preg_grep`.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_grep` returns an empty array (no match), access forbidden.
    $preg_grep = $this->mockPregGrep([]);
    $preg_grep->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      // We still check that other field-level syntax is not affected i.e. the
      // 'edit' operation in the test map is set to `TRUE` so that should be
      // allowed.
      $this->assertEquals(
        TRUE,
        $operation !== 'edit' ? $result->isForbidden() : $result->isNeutral(),
      );
    }
    $preg_grep->disable();
  }

  /**
   * Tests that the result is allowed based on a regular expression when a
   * field-level map item is provided as a string.
   *
   * @covers ::access
   */
  public function testBundleOperationLevelRegexAllowed() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('created');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we mock the result of
    // `preg_grep`.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_grep` returns a non-empty array (one or more roles matched),
    // allowed.
    $preg_grep = $this->mockPregGrep(['anonymous']);
    $preg_grep->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      // We still check that other field-level syntax is not affected i.e. the
      // 'edit' operation in the test map is set to `TRUE` so that should be
      // allowed.
      $this->assertEquals(
        TRUE,
        $operation !== 'create' ? $result->isNeutral() : $result->isForbidden(),
      );
    }
    $preg_grep->disable();
  }

  /**
   * Tests that an error is raised when regex matching fails, usually due to
   * invalid regex passed in a map item.
   *
   * @covers ::access
   */
  public function testBundleOperationLevelRegexInvalid() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we mock the result of
    // `preg_grep`.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_grep` returns a non-empty array (one or more roles matched),
    // allowed.
    $preg_grep = $this->mockPregGrep(FALSE);
    $preg_grep->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $this->expectException(\InvalidArgumentException::class);
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
    }
    $preg_grep->disable();
  }

  /**
   * Tests that the result is always neutral for users with the `administrator`
   * role.
   *
   * @covers ::access
   */
  public function testBundleAdministratorAllowed() {
    // Defined by the default permission map.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['administrator']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // Allow and deny boolean syntax plus ommission at the operation level.
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }

    // Allow boolean syntax at the field level.
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('created');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }

    // Defined by the bundle permission map.
    // Restricted by role and boolean syntax plus ommission at the operation
    // level.
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('body');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }

    // Deny boolean syntax at the field level.
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMap::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
  }

}
