<?php

namespace Drupal\Tests\field_access\Unit;

use Drupal\Core\Field\FieldDefinitionInterface;
use phpmock\Mock;
use phpmock\MockBuilder;
use phpmock\functions\FixedValueFunction;
use Prophecy\Argument\ArgumentsWildcard;
use Prophecy\Prophecy\ObjectProphecy;

trait AccessHandlerTestTrait {

  /**
   * {@inheritdoc}
   *
   * Even though we call the `disable()` method for mocked functions
   * e.g. `$preg_grep->disable();`, it still seems that they are not disabled
   * between tests in some cases e.g. when we catch an expected exception. We
   * therefore make sure we reset any mocked functions between tests.
   */
  public function setUp(): void {
    Mock::disableAll();
  }

  /**
   * Returns a mock object for the field definition.
   */
  protected function prophesizeFieldDefinition($entity_type_id, $bundle) {
    $field_definition = $this->prophesize(FieldDefinitionInterface::class);
    $field_definition
      ->getTargetEntityTypeId()
      ->willReturn($entity_type_id);
    $field_definition
      ->getTargetBundle()
      ->willReturn($bundle);

    return $field_definition;
  }

  /**
   * Returns a function mock for `preg_grep`.
   */
  protected function mockPregGrep($result) {
    $builder = new MockBuilder();
    $builder
      ->setNamespace('\\Drupal\\field_access')
      ->setName('preg_grep')
      ->setFunctionProvider(new FixedValueFunction($result));

    return $builder->build();
  }

  /**
   * Returns a function mock for `preg_match`.
   */
  protected function mockPregMatch($result) {
    $builder = new MockBuilder();
    $builder
      ->setNamespace('\\Drupal\\field_access')
      ->setName('preg_match')
      ->setFunctionProvider(new FixedValueFunction($result));

    return $builder->build();
  }

  /**
   * Asserts that no method has been called on the given prophesized object.
   *
   * @param \Prophecy\Prophecy\ObjectProphecy $prophecy
   *   The prophesized object.
   * @param string $class_name
   *   The class of the original object/interface prophesized.
   */
  protected function assertNoMethodHasBeenCalled(
    ObjectProphecy $prophecy,
    string $class_name,
  ): void {
    $methods = get_class_methods($class_name);

    $calls = [];
    foreach ($methods as $method) {
      $reflection_method = new \ReflectionMethod($class_name, $method);
      $arguments = new ArgumentsWildcard(array_fill(
        0,
        count($reflection_method->getParameters()),
        Argument::any(),
      ));
      $calls = array_merge(
        $calls,
        $prophecy->findProphecyMethodCalls(
          $method,
          $arguments,
        ),
      );
    }

    $this->assertEmpty($calls);
  }

}
