<?php

namespace Drupal\Tests\field_access\Unit;

use Drupal\field_access\AccessHandler;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Tests\UnitTestCase;

use Prophecy\Argument;

/**
 * @coversDefaultClass \Drupal\field_access\AccessHandler
 *
 * @group contrib
 * @group field_access
 * @group unit
 */
class AccessHandlerFieldNameRegexTest extends UnitTestCase {

  use AccessHandlerTestTrait;

  /**
   * Tests that the result is based on a regex pattern when that feature is
   * enabled and when the field name does not have a dedicated map item.
   *
   * @covers ::access
   */
  public function testBundleFieldNameRegexEnabledNoMapItem() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we don't use role syntax
    // for this test.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_match` returns 1 then we have a match; based on the test map
    // used here we should get forbidden, if we didn't have a match we'd get
    // `NULL` map item i.e. neutral.
    $preg_match = $this->mockPregMatch(1);
    $preg_match->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMapRegexEnabled::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isForbidden());
    }
    $preg_match->disable();

    // When `preg_match` returns 0 then we have don't have a match; based on the
    // test map used here we should get neutral, if we had a match we'd get a
    // `FALSE` map item i.e. forbidden.
    $preg_match = $this->mockPregMatch(0);
    $preg_match->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMapRegexEnabled::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
    $preg_match->disable();
  }

  /**
   * Tests that an error is raised when regex matching fails, usually due to
   * invalid regex used as a field name.
   *
   * @covers ::access
   */
  public function testBundleFieldNameRegexEnabledInvalidPattern() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('page');

    // It does not matter what the roles here are since we don't use role syntax
    // for this test.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // When `preg_match` returns `false` then we have a failure i.e. invalid
    // pattern.
    $preg_match = $this->mockPregMatch(false);
    $preg_match->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $this->expectException(\InvalidArgumentException::class);
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMapRegexEnabled::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
    }
    $preg_match->disable();
  }

  /**
   * Tests that the result is NOT based on a regex pattern when that feature is
   * enabled but the field name DOES have a dedicated map item.
   *
   * @covers ::access
   */
  public function testBundleFieldNameRegexEnabledWithMapItem() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('created');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we don't use role syntax
    // for this test.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // What `preg_match` returns in this test should not matter; we therefore
    // return 1 i.e. we have a match so that we can confirm that even if we have
    // a match the regex pattern is not taken into account. We could
    // alternatively assert that `preg_match` is not called at all.
    $preg_match = $this->mockPregMatch(1);
    $preg_match->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMapRegexEnabled::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      $this->assertEquals(TRUE, $result->isNeutral());
    }
    $preg_match->disable();
  }

  /**
   * Tests that the result is NOT based on a regex pattern when that feature is
   * disabled, even when the regex pattern would give us a match and the pattern
   * comes before the field in the map.
   *
   * @covers ::access
   */
  public function testBundleFieldNameRegexDisabledNoMapItem() {
    $field_definition = $this->prophesizeFieldDefinition('node', NULL);
    $field_definition
      ->getName()
      ->willReturn('changed');
    $field_definition
      ->getTargetBundle()
      ->willReturn('post');

    // It does not matter what the roles here are since we don't use role syntax
    // for this test.
    $account = $this->prophesize(AccountInterface::class);
    $account
      ->getRoles()
      ->willReturn(['anonymous']);

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->getDefinition(Argument::any())
      ->shouldNotBeCalled();
    $handler = new AccessHandler($entity_type_manager->reveal());

    // What `preg_match` returns in this test should not matter; we therefore
    // return 1 i.e. we have a match so that we can confirm that even if we have
    // a match the regex pattern is not taken into account. We could
    // alternatively assert that `preg_match` is not called at all.
    $preg_match = $this->mockPregMatch(1);
    $preg_match->enable();
    foreach (['view', 'create', 'edit'] as $operation) {
      $result = $handler->access(
        [
          'node' => NodeTestPermissionMapRegexDisabled::class,
        ],
        $operation,
        $field_definition->reveal(),
        $account->reveal(),
        NULL,
      );
      // If the regex pattern was used we'd get forbidden.
      $this->assertEquals(TRUE, $result->isNeutral());
    }
    $preg_match->disable();
  }

}
