/**
 * @file
 * Field Group Complete — tab label indicator (class-driven, optional badge).
 *
 * Settings (set via drupalSettings.field_group_complete):
 *   selectors: ['.field-group-tab', ...]             // required (from PHP)
 *   badgeText: { complete: 'Complete', incomplete: 'Incomplete' }
 *   customCompleteClasses: ['my-complete-class']     // optional, applied to tab when complete
 *   badgeVisibility: 'visible' | 'sr-only' | 'none'  // default 'visible'
 */
((Drupal, drupalSettings, once) => {
  const SETTINGS = drupalSettings?.field_group_complete || {};
  const BADGE_VIS = (['visible', 'sr-only', 'none'].includes(SETTINGS.badgeVisibility)) ? SETTINGS.badgeVisibility : 'visible';

  // ---------------- helpers: visibility/required/value ----------------
  function isVisible(el) {
    if (!(el instanceof Element) || el.hidden) return false;
    const style = getComputedStyle(el);
    if (style.display === 'none' || style.visibility === 'hidden' || style.opacity === '0') return false;
    if (!el.getClientRects().length) return false;
    return true;
  }
  const isEnabled = (el) => !el.disabled;

  function hasRequiredMarkerOnLabel(el) {
    if (!el.id) return false;
    const lbl = el.ownerDocument.querySelector('label[for="' + CSS.escape(el.id) + '"]');
    return !!(lbl && (lbl.classList.contains('js-form-required') || lbl.classList.contains('form-required')));
  }
  function isWithinRequiredWrapper(el) {
    const wrapper = el.closest('.js-form-item, .form-item, fieldset.form-composite');
    if (!wrapper) return false;
    const legend = wrapper.querySelector('legend, .fieldset-legend');
    if (legend && (legend.classList.contains('js-form-required') || legend.classList.contains('form-required'))) return true;
    return wrapper.classList.contains('js-form-required') || wrapper.classList.contains('form-required');
  }
  function isRequired(el) {
    return el.hasAttribute('required') ||
      el.getAttribute('aria-required') === 'true' ||
      el.classList.contains('required') ||
      hasRequiredMarkerOnLabel(el) ||
      isWithinRequiredWrapper(el);
  }

  function hasValue(el, scope) {
    const type = (el.getAttribute('type') || el.tagName).toLowerCase();
    if (type === 'checkbox') return el.checked;
    if (type === 'radio') {
      const name = el.getAttribute('name');
      if (!name) return false;
      const radios = scope.querySelectorAll('input[type="radio"][name="' + CSS.escape(name) + '"]');
      for (const r of radios) if (r.checked) return true;
      return false;
    }
    if (el.tagName === 'SELECT') return el.value !== '' && el.value !== null;
    return (el.value || '').trim().length > 0;
  }

  function computeComplete(groupEl) {
    const controls = groupEl.querySelectorAll('input, select, textarea');
    let req = 0, ok = 0;
    for (const el of controls) {
      const vis = isVisible(el);
      const ena = isEnabled(el);
      const reqd = isRequired(el);
      const val = reqd && vis && ena ? hasValue(el, groupEl) : false;
      if (!vis || !ena || !reqd) continue;
      req++; if (val) ok++;
    }
    const complete = (req === 0) ? true : (ok === req);
    return complete;
  }

  // ---------------- tab mapping + UI ---------------------------------
  function findTabMenuAnchorForPane(paneEl) {
    if (!paneEl.id) return null;
    const hash = '#' + paneEl.id;
    const vt = paneEl.closest('.vertical-tabs');
    if (vt) {
      const a = vt.querySelector('.vertical-tabs__menu a[href="' + CSS.escape(hash) + '"]');
      if (a) return a;
    }
    const ha = paneEl.ownerDocument.querySelector('.horizontal-tabs-list a[href="' + CSS.escape(hash) + '"]');
    if (ha) return ha;
    return paneEl.ownerDocument.querySelector('a[href="' + CSS.escape(hash) + '"]');
  }

  function applyCustomClasses(targetEl, on) {
    const extra = Array.isArray(SETTINGS.customCompleteClasses) ? SETTINGS.customCompleteClasses : [];
    for (const cls of extra) {
      const c = String(cls || '').trim();
      if (!c) continue;
      if (on) targetEl.classList.add(c); else targetEl.classList.remove(c);
    }
  }

  function ensureBadgeOnTab(tabAnchor) {
    if (BADGE_VIS === 'none') return null;
    const content = tabAnchor.querySelector('.vertical-tabs__menu-link-content') || tabAnchor;
    let badge = content.querySelector('.field-group-complete__badge');
    if (!badge) {
      badge = document.createElement('span');
      badge.className = 'field-group-complete__badge';
      badge.setAttribute('aria-live', 'polite');

      const text = document.createElement('span');
      text.className = 'field-group-complete__badge-text';
      text.textContent = SETTINGS?.badgeText?.incomplete || Drupal.t('Incomplete');

      badge.appendChild(text);
      content.appendChild(badge);
    }
    // a11y option: visually hide but keep SR announcement.
    if (BADGE_VIS === 'sr-only') {
      badge.classList.add('visually-hidden');
    } else {
      badge.classList.remove('visually-hidden');
    }
    return badge;
  }

  function updateBadgeText(badge, complete) {
    if (!badge) return;
    const text = badge.querySelector('.field-group-complete__badge-text');
    if (text) {
      text.textContent = complete
        ? (SETTINGS?.badgeText?.complete || Drupal.t('Complete'))
        : (SETTINGS?.badgeText?.incomplete || Drupal.t('Incomplete'));
    }
  }

  function setState(paneEl, tabAnchor, complete) {
    // Pane hooks (for theming if desired).
    paneEl.classList.add('field-group-complete--enhanced');
    paneEl.toggleAttribute('data-field-group-complete', !!complete);
    paneEl.classList.toggle('field-group-complete--complete', !!complete);

    // Tab-level classes = primary contract.
    const tabTarget = tabAnchor.closest('li') || tabAnchor;
    tabTarget.classList.add('field-group-complete--enhanced');
    tabTarget.classList.toggle('field-group-complete--complete', !!complete);
    applyCustomClasses(tabTarget, complete);
  }

  // ---------------- behavior -----------------------------------------
  function enhanceGroup(paneEl) {
    const tabAnchor = findTabMenuAnchorForPane(paneEl);

    let raf = 0, last = null;
    const recompute = () => {
      if (raf) cancelAnimationFrame(raf);
      raf = requestAnimationFrame(() => {
        const badge = ensureBadgeOnTab(tabAnchor); // creates/updates visibility or returns null
        const cur = computeComplete(paneEl);
        if (cur !== last) {
          last = cur;
          setState(paneEl, tabAnchor, cur);
          updateBadgeText(badge, cur);
          tabAnchor.dispatchEvent(new CustomEvent('field-group-complete:update', {
            bubbles: false, cancelable: false, detail: { paneEl, tabAnchor, complete: cur }
          }));
        } else {
          updateBadgeText(badge, cur);
        }
      });
    };

    paneEl.addEventListener('input',  recompute, true);
    paneEl.addEventListener('change', recompute, true);

    const mo = new MutationObserver(recompute);
    mo.observe(paneEl, {
      subtree: true, childList: true, attributes: true,
      attributeFilter: ['required', 'aria-required', 'disabled', 'class', 'style', 'hidden'],
    });

    setTimeout(recompute, 0);
  }

  Drupal.behaviors.fieldGroupComplete = {
    attach(context) {
      const selectors = SETTINGS.selectors || [];
      if (!Array.isArray(selectors) || !selectors.length) return;
      const panes = once('field-group-complete', selectors.join(','), context);
      if (!panes.length) return;
      setTimeout(() => panes.forEach(enhanceGroup), 0);
    }
  };
})(Drupal, drupalSettings, once);
