<?php

namespace Drupal\field_group_complete\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Field Group Complete.
 */
class SettingsForm extends ConfigFormBase {

  const SETTINGS = 'field_group_complete.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'field_group_complete_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [static::SETTINGS];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);

    $form['custom_complete_classes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Extra complete-classes'),
      '#description' => $this->t('One CSS class per line to add when a group is complete. Useful for design-system integrations.'),
      '#default_value' => implode("\n", (array) $config->get('custom_complete_classes') ?: []),
    ];

    $form['badge_text_complete'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Badge text (complete)'),
      '#default_value' => (string) $config->get('badge_text_complete') ?: 'Complete',
    ];
    $form['badge_text_incomplete'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Badge text (incomplete)'),
      '#default_value' => (string) $config->get('badge_text_incomplete') ?: 'Incomplete',
    ];

    $form['badge_visibility'] = [
      '#type' => 'select',
      '#title' => $this->t('Badge visibility'),
      '#description' => $this->t('Control how the completion badge appears on tab labels.'),
      '#options' => [
        'visible' => $this->t('Visible (default)'),
        'sr-only' => $this->t('Screen reader only (visually hidden)'),
        'none' => $this->t('No badge (class-only mode)'),
      ],
      '#default_value' => (string) $config->get('badge_visibility') ?: 'visible',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $extra = preg_split('/\r?\n/', (string) $form_state->getValue('custom_complete_classes'), -1, PREG_SPLIT_NO_EMPTY);
    $extra = array_values(array_unique(array_map('trim', $extra)));

    $this->configFactory()->getEditable(static::SETTINGS)
      ->set('custom_complete_classes', $extra)
      ->set('badge_text_complete', (string) $form_state->getValue('badge_text_complete'))
      ->set('badge_text_incomplete', (string) $form_state->getValue('badge_text_incomplete'))
      ->set('badge_visibility', (string) $form_state->getValue('badge_visibility'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
