<?php

declare(strict_types=1);

namespace Drupal\Tests\field_group_complete\FunctionalJavascript;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Verifies tab label completeness indicator updates live (Tabs → Tab only).
 *
 * @group field_group_complete
 */
final class TabBadgeTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'text',
    'field_ui',
    'field_group',
    'field_group_complete',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create bundle FIRST so per-bundle perms exist.
    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);

    // Create a required field on Article.
    if (!FieldStorageConfig::loadByName('node', 'field_favorite_fruit')) {
      FieldStorageConfig::create([
        'entity_type' => 'node',
        'field_name' => 'field_favorite_fruit',
        'type' => 'string',
        'cardinality' => 1,
        'settings' => ['max_length' => 255],
      ])->save();
    }
    if (!FieldConfig::loadByName('node', 'article', 'field_favorite_fruit')) {
      FieldConfig::create([
        'entity_type' => 'node',
        'bundle' => 'article',
        'field_name' => 'field_favorite_fruit',
        'label' => 'Favorite Fruit',
        'required' => TRUE,
      ])->save();
    }

    // Put the field on the default form display.
    $form_display = \Drupal::entityTypeManager()
      ->getStorage('entity_form_display')
      ->load('node.article.default');
    if (!$form_display) {
      $form_display = \Drupal::entityTypeManager()
        ->getStorage('entity_form_display')
        ->create([
          'targetEntityType' => 'node',
          'bundle' => 'article',
          'mode' => 'default',
          'status' => TRUE,
        ]);
    }
    $form_display->setComponent('field_favorite_fruit', [
      'type' => 'string_textfield',
      'weight' => 0,
      'region' => 'content',
    ]);

    // Define Field Group: Tabs container + one Tab leaf via third-party
    // settings. Field Group stores its groups directly under
    // third_party_settings[field_group][GROUP_ID].
    $form_display->setThirdPartySetting('field_group', 'group_page_tabs', [
      'children' => ['group_favorites'],
      'parent_name' => '',
      'weight' => -10,
      // Container plugin.
      'format_type' => 'tabs',
      'format_settings' => [
        'id' => '',
        'classes' => '',
        'direction' => 'vertical',
      ],
      'label' => 'Page Tabs',
      'region' => 'content',
    ]);

    $form_display->setThirdPartySetting('field_group', 'group_favorites', [
      'children' => ['field_favorite_fruit'],
      'parent_name' => 'group_page_tabs',
      'weight' => 0,
      // Leaf plugin.
      'format_type' => 'tab',
      'format_settings' => [
        'id' => '',
        'classes' => '',
        'formatter' => 'open',
      ],
      'label' => 'Favorites',
      'region' => 'content',
    ]);

    $form_display->save();

    // 5) Create and log in a user.
    $editor = $this->drupalCreateUser([
      'access content',
      'create article content',
    ]);
    $this->drupalLogin($editor);

    // 6) Ensure render caches are rebuilt so Field Group markup appears.
    drupal_flush_all_caches();
  }

  /**
   * Test toggle complete indicator.
   */
  public function testFieldGroupCompleteToggle(): void {
    $this->drupalGet('node/add/article');
    $session = $this->assertSession();

    // Wait for Field Group vertical tabs to appear.
    $session->waitForElement('css', '.vertical-tabs__menu a[href="#edit-group-favorites"]');

    // Initially, tab should NOT have the complete class.
    $tab_link = $this->getSession()->getPage()->find('css', '.vertical-tabs__menu a[href="#edit-group-favorites"]');
    $this->assertNotNull($tab_link, 'Found tab link for Favorites.');
    $tab_li = $tab_link->getParent();
    $this->assertNotNull($tab_li, 'Found tab list item.');
    $this->assertFalse(
      str_contains($tab_li->getAttribute('class') ?? '', 'field-group-complete--complete'),
      'Tab initially not complete.'
    );

    // Fill the required field inside the tab.
    $session->waitForElement('css', '#edit-group-favorites');
    $this->getSession()->getPage()->fillField('Favorite Fruit', 'Banana');

    // Wait until the *real* pane (<details>) gets the complete class.
    $session->waitForElement(
      'css',
      'details.field-group-tab.field-group-complete--complete'
    );

    // Re-fetch tab <li> and assert class.
    $tab_link = $this->getSession()->getPage()->find('css', '.vertical-tabs__menu a[href="#edit-group-favorites"]');
    $this->assertNotNull($tab_link);
    $tab_li = $tab_link->getParent();
    $this->assertTrue(
      str_contains($tab_li->getAttribute('class') ?? '', 'field-group-complete--complete'),
      'Tab is marked complete after input.'
    );

    // If the badge is present (badgeVisibility != "none"), assert its text.
    $badge = $this->getSession()->getPage()->find('css', '.vertical-tabs__menu a[href="#edit-group-favorites"] .field-group-complete__badge-text');
    if ($badge) {
      $this->assertEquals('Complete', trim($badge->getText()));
    }
  }

}
