<?php

namespace Drupal\field_inheritance\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the Field inheritance entity.
 *
 * @ConfigEntityType(
 *   id = "field_inheritance",
 *   label = @Translation("Field inheritance"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\field_inheritance\FieldInheritanceListBuilder",
 *     "form" = {
 *       "add" = "Drupal\field_inheritance\Form\FieldInheritanceForm",
 *       "edit" = "Drupal\field_inheritance\Form\FieldInheritanceForm",
 *       "delete" = "Drupal\field_inheritance\Form\FieldInheritanceDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\field_inheritance\FieldInheritanceHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "field_inheritance",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/field_inheritance/{field_inheritance}",
 *     "add-form" = "/admin/structure/field_inheritance/add",
 *     "edit-form" = "/admin/structure/field_inheritance/{field_inheritance}/edit",
 *     "delete-form" = "/admin/structure/field_inheritance/{field_inheritance}/delete",
 *     "collection" = "/admin/structure/field_inheritance"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "type",
 *     "sourceEntityType",
 *     "destinationEntityType",
 *     "sourceEntityBundle",
 *     "destinationEntityBundle",
 *     "sourceField",
 *     "destinationField",
 *     "plugin"
 *   }
 * )
 */
class FieldInheritance extends ConfigEntityBase implements FieldInheritanceInterface {

  /**
   * The field inheritance ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The field inheritance label.
   *
   * @var string
   */
  protected $label;

  /**
   * The field inheritance type.
   *
   * @var string
   */
  protected $type;

  /**
   * The field inheritance source entity type.
   *
   * @var string
   */
  protected $sourceEntityType;

  /**
   * The field inheritance destination entity type.
   *
   * @var string
   */
  protected $destinationEntityType;

  /**
   * The field inheritance source entity bundle.
   *
   * @var string
   */
  protected $sourceEntityBundle;

  /**
   * The field inheritance destination entity bundle.
   *
   * @var string
   */
  protected $destinationEntityBundle;

  /**
   * The field inheritance source field.
   *
   * @var string
   */
  protected $sourceField;

  /**
   * The field inheritance destination field.
   *
   * @var string
   */
  protected $destinationField;

  /**
   * The field inheritance plugin.
   *
   * @var string
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  public function type(): string|null {
    return $this->type ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function idWithoutTypeAndBundle(): string|null {
    $prefix = $this->destinationEntityType() . '_' . $this->destinationEntityBundle() . '_';
    return str_replace($prefix, '', $this->id());
  }

  /**
   * {@inheritdoc}
   */
  public function sourceEntityType(): string|null {
    return $this->sourceEntityType ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function destinationEntityType(): string|null {
    return $this->destinationEntityType ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function sourceEntityBundle(): string|null {
    return $this->sourceEntityBundle ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function destinationEntityBundle(): string|null {
    return $this->destinationEntityBundle ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function sourceField(): string|null {
    return $this->sourceField ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function destinationField(): string|null {
    return $this->destinationField ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function sourceIdentifier(): string {
    return $this->sourceEntityType() . ':' . $this->sourceEntityBundle();
  }

  /**
   * {@inheritdoc}
   */
  public function destinationIdentifier(): string {
    return $this->destinationEntityType() . ':' . $this->destinationEntityBundle();
  }

  /**
   * {@inheritdoc}
   */
  public function plugin(): string|null {
    return $this->plugin ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setType(string $type): static {
    $this->type = $type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setSourceEntityType(string $source_entity_type): static {
    $this->sourceEntityType = $source_entity_type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setDestinationEntityType(string $destination_entity_type): static {
    $this->destinationEntityType = $destination_entity_type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setSourceEntityBundle(string $source_entity_bundle): static {
    $this->sourceEntityBundle = $source_entity_bundle;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setDestinationEntityBundle(string $destination_entity_bundle): static {
    $this->destinationEntityBundle = $destination_entity_bundle;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setSourceField(string $source_field): static {
    $this->sourceField = $source_field;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setDestinationField(string $destination_field): static {
    $this->destinationField = $destination_field;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setPlugin(string $plugin): static {
    $this->plugin = $plugin;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function save(): int {
    if (!str_contains($this->id(), $this->destinationEntityType() . '_' . $this->destinationEntityBundle() . '_')) {
      $this->id = $this->destinationEntityType() . '_' . $this->destinationEntityBundle() . '_' . $this->id();
    }
    return parent::save();
  }

}
