<?php

namespace Drupal\field_inheritance\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Provides an interface for defining Field inheritance entities.
 */
interface FieldInheritanceInterface extends ConfigEntityInterface {

  /**
   * Get the inheritance type.
   *
   * @return string|null
   *   The inheritance type.
   */
  public function type(): string|null;

  /**
   * Get the inheritance ID without the type and bundle.
   *
   * @return string|null
   *   The inheritance ID without the type and bundle.
   */
  public function idWithoutTypeAndBundle(): string|null;

  /**
   * Get the inheritance source entity type.
   *
   * @return string|null
   *   The inheritance source entity type.
   */
  public function sourceEntityType(): string|null;

  /**
   * Get the inheritance destination entity type.
   *
   * @return string|null
   *   The inheritance destination entity type.
   */
  public function destinationEntityType(): string|null;

  /**
   * Get the inheritance source entity bundle.
   *
   * @return string|null
   *   The inheritance source entity bundle.
   */
  public function sourceEntityBundle(): string|null;

  /**
   * Get the inheritance destination entity bundle.
   *
   * @return string|null
   *   The inheritance destination entity bundle.
   */
  public function destinationEntityBundle(): string|null;

  /**
   * Get the inheritance source field.
   *
   * @return string|null
   *   The inheritance source field.
   */
  public function sourceField(): string|null;

  /**
   * Get the inheritance destination field.
   *
   * @return string|null
   *   The inheritance destination field.
   */
  public function destinationField(): string|null;

  /**
   * Get the inheritance source identifier.
   *
   * @return string
   *   The inheritance source identifier.
   */
  public function sourceIdentifier(): string;

  /**
   * Get the inheritance destination identifier.
   *
   * @return string
   *   The inheritance destination identifier.
   */
  public function destinationIdentifier(): string;

  /**
   * Get the inheritance plugin.
   *
   * @return string|null
   *   The inheritance plugin.
   */
  public function plugin(): string|null;

  /**
   * Set the inheritance type.
   *
   * @var string $type
   *   The inheritance type.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setType(string $type): static;

  /**
   * Set the inheritance source entity type.
   *
   * @var string $source_entity_type
   *   The inheritance source entity type.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setSourceEntityType(string $source_entity_type): static;

  /**
   * Set the inheritance destination entity type.
   *
   * @var string $destination_entity_type
   *   The inheritance destination entity type.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setDestinationEntityType(string $destination_entity_type): static;

  /**
   * Set the inheritance source entity bundles.
   *
   * @var string $source_entity_bundle
   *   The inheritance source entity bundle.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setSourceEntityBundle(string $source_entity_bundle): static;

  /**
   * Set the inheritance destination entity bundles.
   *
   * @var string $destination_entity_bundle
   *   The inheritance destination entity bundle.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setDestinationEntityBundle(string $destination_entity_bundle): static;

  /**
   * Set the inheritance source field.
   *
   * @var string $source_field
   *   The inheritance source field.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setSourceField(string $source_field): static;

  /**
   * Set the inheritance destination field.
   *
   * @var string $destination_field
   *   The inheritance destination field.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setDestinationField(string $destination_field): static;

  /**
   * Set the inheritance plugin.
   *
   * @var string $plugin
   *   The inheritance plugin.
   *
   * @return static
   *   The FieldInheritanceInterface object.
   */
  public function setPlugin(string $plugin): static;

}
