<?php

declare(strict_types=1);

namespace Drupal\field_inheritance\EventSubscriber;

use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Config events subscriber for field_inheritance.
 */
final class ConfigSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructs a ConfigSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface $entityDefinitionUpdateManager
   *   The entity definition update manager.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly EntityDefinitionUpdateManagerInterface $entityDefinitionUpdateManager,
  ) {}

  /**
   * Reacts to Config changes to manage the related base fields.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The config event.
   */
  public function onConfigSave(ConfigCrudEvent $event): void {
    $config = $event->getConfig();
    if ($config->getName() !== 'field_inheritance.config') {
      return;
    }

    // The is_array is needed for the 2.x to 3.x upgrade path. Could be
    // simplified in the future.
    $original_entity_types = is_array($config->getOriginal('included_entities')) ?
      $config->getOriginal('included_entities') : [];
    $entity_types = is_array($config->get('included_entities')) ?
      $config->get('included_entities') : [];

    foreach (array_diff($entity_types, $original_entity_types) as $entity_type_id) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);
      if (!$entity_type || in_array('field_inheritance', $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id))) {
        continue;
      }
      $definition = BaseFieldDefinition::create('map')
        ->setLabel($this->t('Field Inheritance'))
        ->setDescription($this->t('A serialized array of order options.'))
        ->setRevisionable($entity_type->isRevisionable())
        ->setDefaultValue(serialize([]))
        ->setDisplayConfigurable('form', TRUE);

      $this->entityDefinitionUpdateManager->installFieldStorageDefinition('field_inheritance', $entity_type_id, 'field_inheritance', $definition);
      $this->entityTypeManager->getStorage($entity_type_id)->resetCache();
    }

    foreach (array_diff($original_entity_types, $entity_types) as $entity_type_id) {
      $definition = $this->entityDefinitionUpdateManager->getFieldStorageDefinition('field_inheritance', $entity_type_id);
      try {
        $this->entityDefinitionUpdateManager->uninstallFieldStorageDefinition($definition);
        $this->entityTypeManager->getStorage($entity_type_id)->resetCache();
      }
      catch (EntityStorageException $e) {
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConfigEvents::SAVE => ['onConfigSave'],
    ];
  }

}
