<?php

declare(strict_types=1);

namespace Drupal\field_inheritance\EventSubscriber;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldStorageDefinitionEvent;
use Drupal\Core\Field\FieldStorageDefinitionEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * FieldStorageDefinitionSubscriber implementation.
 *
 * This subscriber is a workaround for
 * https://www.drupal.org/project/drupal/issues/3300404
 *
 * We could remove it once the issue is fixed.
 */
final class FieldStorageDefinitionSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a FieldStorageDefinitionSubscriber object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   */
  public function __construct(
    private readonly Connection $database,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
  ) {}

  /**
   * Kernel request event handler.
   */
  public function onFieldStorageDefinitionCreate(FieldStorageDefinitionEvent $event, $event_name): void {
    $entity_type_id = $event->getFieldStorageDefinition()->getTargetEntityTypeId();
    $base_fields = $this->entityFieldManager->getBaseFieldDefinitions($entity_type_id);
    if (!array_key_exists('field_inheritance', $base_fields)) {
      return;
    }
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $data_table = $entity_type->getDataTable() ?? $entity_type->getBaseTable();
    if ($data_table) {
      $this->database->update($data_table)
        ->fields(['field_inheritance' => serialize([])])
        ->isNull('field_inheritance')
        ->execute();
    }
    $revision_data_table = $entity_type->getRevisionDataTable() ?? $entity_type->getRevisionTable();
    if ($revision_data_table) {
      $this->database->update($revision_data_table)
        ->fields(['field_inheritance' => serialize([])])
        ->isNull('field_inheritance')
        ->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      FieldStorageDefinitionEvents::CREATE => ['onFieldStorageDefinitionCreate'],
    ];
  }

}
