<?php

declare(strict_types=1);

namespace Drupal\field_inheritance\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Builds a confirmation form to remove entity types.
 */
class FieldInheritanceSettingsConfirmForm extends ConfirmFormBase {

  use AutowireTrait;

  /**
   * The entity types.
   *
   * @var array
   */
  protected array $entityTypes = [];

  /**
   * Constructs a new FieldInheritanceSettingsConfirmForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface $keyValueExpirable
   *   The keyvalue expirable.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'keyvalue.expirable')]
    protected KeyValueExpirableFactoryInterface $keyValueExpirable,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'field_inheritance_settings_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Confirm entity removal');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Save configuration');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('field_inheritance.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('Would you like to continue with removing the above?');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array|RedirectResponse {
    // Retrieve the list of entity types from the key value store.
    $account = $this->currentUser()->id();
    $this->entityTypes = $this->keyValueExpirable->get('field_inheritance_to_remove')->get($account);

    // Prevent this page from showing when the module list is empty.
    if (empty($this->entityTypes)) {
      return $this->redirect('system.modules_uninstall');
    }

    $original_entity_types = $this->config('field_inheritance.config')->getOriginal('included_entities');

    $to_remove = array_diff($original_entity_types, $this->entityTypes);

    $form['text']['#markup'] = '<p>' . $this->t('Field inheritance will be disabled for the following entity types and <em>all the Field Inheritance data from these field types will be lost</em>!') . '</p>';
    $form['entity_types'] = [
      '#theme' => 'item_list',
      '#items' => array_map(function (string $entity_type_id) {
        return $this->entityTypeManager->getDefinition($entity_type_id, FALSE)?->getLabel() ?? $entity_type_id;
      }, $to_remove),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Clear the key value store entry.
    $account = $this->currentUser()->id();
    $this->keyValueExpirable->get('field_inheritance_to_remove')->delete($account);

    // Uninstall the modules.
    $this->configFactory()->getEditable('field_inheritance.config')
      ->set('included_entities', $this->entityTypes)
      ->save();

    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
